#version 130

// Shader by LoLip_p

#include "distort.glsl"
#include "/settings.glsl"

varying vec2 TexCoords;

uniform float rainStrength;
uniform int worldTime;

uniform sampler2D colortex0;
uniform sampler2D colortex1;
uniform sampler2D colortex2;
uniform sampler2D depthtex0;

// For Shadow
uniform sampler2D shadowtex0;
uniform sampler2D shadowtex1;
uniform sampler2D shadowcolor0;
uniform sampler2D noisetex;
uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferModelViewInverse;
uniform mat4 shadowModelView;
uniform mat4 shadowProjection;

/*
const int colortex0Format = RGB16;
const int colortex1Format = RGB8;
const int colortex2Format = RGB8;
const int colortex3Format = RGB8;
const int colortex4Format = RGB16;
const int colortex5Format = RGB8;
*/

const int noiseTextureResolution = 64;

float state;

float Visibility(in sampler2D ShadowMap, in vec3 SampleCoords) {
    return step(SampleCoords.z - 0.0002f, texture2D(ShadowMap, SampleCoords.xy).r);
}

vec3 TransparentShadow(in vec3 SampleCoords){
    float ShadowVisibility0 = Visibility(shadowtex0, SampleCoords);
    float ShadowVisibility1 = Visibility(shadowtex1, SampleCoords);
    vec4 ShadowColor0 = texture2D(shadowcolor0, SampleCoords.xy);
    vec3 TransmittedColor = ShadowColor0.rgb * (1.0f - ShadowColor0.a); // Perform a blend operation with the sun color
    return mix(TransmittedColor * ShadowVisibility1, vec3(1.0f), ShadowVisibility0);
}

#define SHADOW_SAMPLES 1
const int ShadowSamplesPerSize = 2 * SHADOW_SAMPLES + 1;
const int TotalSamples = ShadowSamplesPerSize * ShadowSamplesPerSize;

vec3 GetShadow(float depth) {
    vec3 ClipSpace = vec3(TexCoords, depth) * 2.0f - 1.0f;
	
    vec4 ViewW = gbufferProjectionInverse * vec4(ClipSpace, 1.0f);
    vec3 View = ViewW.xyz / ViewW.w;
	
    vec4 World = gbufferModelViewInverse * vec4(View, 1.0f);
    vec4 ShadowSpace = shadowProjection * shadowModelView * World;
    ShadowSpace.xy = DistortPosition(ShadowSpace.xy);
    vec3 SampleCoords = ShadowSpace.xyz * 0.5f + 0.5f;
	
	SampleCoords = clamp(SampleCoords, 0.0, 1.0); // clamp
	
    float RandomAngle = texture2D(noisetex, TexCoords * 20.0f).r * 100.0f;
    float cosTheta = cos(RandomAngle);
	float sinTheta = sin(RandomAngle);
    mat2 Rotation =  mat2(cosTheta, -sinTheta, sinTheta, cosTheta) / shadowMapResolution; // We can move our division by the shadow map resolution here for a small speedup
    vec3 ShadowAccum = vec3(0.0f);
    for(int x = -SHADOW_SAMPLES; x <= SHADOW_SAMPLES; x++){
        for(int y = -SHADOW_SAMPLES; y <= SHADOW_SAMPLES; y++){
            vec2 Offset = Rotation * vec2(x, y);
            vec3 CurrentSampleCoordinate = vec3(SampleCoords.xy + Offset, SampleCoords.z);
			
			CurrentSampleCoordinate = clamp(CurrentSampleCoordinate, 0.0, 1.0); // clamp
			
            ShadowAccum += TransparentShadow(CurrentSampleCoordinate);
        }
    }
    ShadowAccum /= TotalSamples;
    return ShadowAccum;
}

float AdjustLightmapTorch(in float torchLight) {
    const float K = 2.0f;
	const float P = 5.0f;
    return K * pow(torchLight, P);
}

float AdjustLightmapSky(in float sky){
    float sky_2 = sky * sky;
    return sky_2 * sky_2;
}

vec2 AdjustLightmap(in vec2 Lightmap){
    vec2 NewLightMap;
    NewLightMap.x = AdjustLightmapTorch(Lightmap.x);
    NewLightMap.y = AdjustLightmapSky(Lightmap.y);
    return NewLightMap;
}

vec3 GetLightmapColor(in vec2 Lightmap){
    Lightmap = AdjustLightmap(Lightmap);
	
	#if TI == 0
    const vec3 SkyColor = vec3(0.05f, 0.15f, 0.3f);
    const vec3 TorchColor = vec3(1.0f, 0.45f, 0.3f);
	#else
	const vec3 SkyColor = vec3(0);
	const vec3 TorchColor = vec3(1);
	#endif
	
	
    vec3 TorchLighting = Lightmap.x * vec3(1.0f, 1.0f, 1.0f);
    vec3 SkyLighting = Lightmap.y * SkyColor * state * (1 - rainStrength);
	
	#if TI == 0
	TorchLighting -= (Lightmap.y * LIGHT_ABSORPTION * state * (1 - rainStrength));
	#endif
	
	vec3 LightmapLighting = clamp(TorchLighting * TorchColor, 0.0f, 1.0f) + SkyLighting;
	
    return LightmapLighting;
}

float smoothTransition(float time) {
    if (time >= 0.0f && time <= 1000.0f) {
        return time / 1000.0f; // Transition from 0 to 1
    } else if (time > 1000.0f && time < 12000.0f) {
        return 1.0f; // Fully enabled
    } else if (time >= 12000.0f && time <= 13000.0f) {
        return 1.0f - (time - 12000.0f) / 1000.0f; // Transition from 1 to 0
    } else {
        return 0.0f; // Fully disabled
    }
}

float swapDayNight(float time) {
    if (time >= 12300.0f && time <= 12800.0f) {
        return (time - 12300.0f) / 500.0f; // Плавный переход от 0 до 1
    } else if (time > 12800.0f && time <= 13200.0f) {
        return 1.0f - (time - 12800.0f) / 500.0f; // Плавный переход от 1 до 0
    } else if (time >= 22700.0f && time <= 23200.0f) {
        return (time - 22700.0f) / 500.0f; // Плавный переход от 0 до 1
    } else if (time > 23200.0f && time <= 23700.0f) {
        return 1.0f - (time - 23200.0f) / 500.0f; // Плавный переход от 1 до 0
    } else {
        return 0.0f; // Вне указанных диапазонов
    }
}

void main(){
	vec3 Albedo = pow(texture2D(colortex0, TexCoords).rgb, vec3(2.2f));

    vec3 ShadowColor = vec3(1);
	vec3 NdotL = vec3(1);
	
	vec2 Lightmap = texture2D(colortex2, TexCoords).rg;
	state = smoothTransition(worldTime);
	vec3 LightmapColor = GetLightmapColor(Lightmap);
	
    float Depth = texture2D(depthtex0, TexCoords).r;
    if(Depth == 1.0f){
	    vec3 colorSky = Albedo;
	
        gl_FragData[0] = vec4(colorSky, 1.0f);
        return;
    }

	#if SHADING == 1
    vec3 Normal = texture2D(colortex1, TexCoords).rgb * 2.0f - 1.0f;
    NdotL = (vec3(0.0) * (1.0f - rainStrength) * 0.75f + SHADOW_TRANSPARENCY) * state; //Day
	NdotL += vec3(0.2, 0.2, 0.5) * (1.0f - state); //Night
	#endif

	NdotL *= AdjustLightmap(Lightmap).g;
	NdotL += 0.025f;
	
    #if ENABLE_SHADOW == 1
    vec3 originalShadow = GetShadow(Depth);
	float shadow_transperency_coef = mix(degree_night_darkness, SHADOW_TRANSPARENCY, state);
	ShadowColor = mix(originalShadow * vec3(shadow_transperency_coef / 0.75), originalShadow, smoothTransition(worldTime));
	ShadowColor = mix(ShadowColor, vec3(1), swapDayNight(worldTime));
	vec3 colorRainDayOrNight = mix(vec3(0.2), vec3(1), state);
	ShadowColor = mix(ShadowColor, colorRainDayOrNight, rainStrength);
	
	ShadowColor *= AdjustLightmap(Lightmap).g;
	ShadowColor += shadow_transperency_coef;
	#endif
	
	vec3 Diffuse = Albedo * (LightmapColor + NdotL * ShadowColor);
	
	/* DRAWBUFFERS:0 */
    gl_FragData[0] = vec4(Diffuse, 1.0f);
}
