#include "/lib/core.glsl"
#include "/lib/config.glsl"

uniform vec3 shadowLightDirection;
uniform mat3 normalMatrix;
uniform mat4 gbufferModelViewInverse, modelViewMatrix, projectionMatrix, shadowModelView, shadowProjection, textureMatrix;

#include "/lib/fast_math.glsl"
#include "/lib/distort.glsl"
#include "/lib/vertex_block_light.glsl"

#ifndef NO_NORMAL
	in vec3 vaNormal;
	in vec4 at_tangent;
#endif

#if defined TERRAIN || !(defined NM && defined MC_NORMAL_MAP)
	in vec2 mc_midTexCoord;
#endif

#ifdef TERRAIN
	uniform ivec2 atlasSize;
	uniform vec3 chunkOffset;

	in vec2 mc_Entity;
	in vec4 at_midBlock;

	#if WAVES
		#include "/lib/waves/offset.glsl"
	#endif
#endif

#if !defined TERRAIN || !(SM && defined MC_SPECULAR_MAP)
	uniform sampler2D gtexture;
#endif

#ifdef ENTITY_COLOR
	uniform vec4 entityColor;
#endif

in vec2 vaUV0;
in vec3 vaPosition;
in vec4 vaColor;

out VertexData {
	vec2 coord;
	vec3 view;

	#ifndef NETHER
		vec3 s_ndc;
	#endif

	#ifndef NO_NORMAL
		flat mat3 tbn;
	#endif

	#if !(defined NM && defined MC_NORMAL_MAP)
		flat uint texels;
		flat vec2 mid_coord;
	#endif

	#ifdef TERRAIN
		vec4 color;

		#if !(SM && defined MC_SPECULAR_MAP)
			flat float avg_luma;
		#endif
	#else
		vec3 color;
	#endif
} f;

#include "/lib/srgb.glsl"

void main() {
	immut vec2 coord = (textureMatrix * vec4(vaUV0, 0.0, 1.0)).xy;
	vec3 model = vaPosition;

	#ifdef TERRAIN
		vec4 color = vec4(vaColor.rgb, 1.0);

		model += chunkOffset;

		if (mc_Entity.y > 0.5) {
			color.a *= WATER_OPACITY * 0.01;

			#if WAVES
				model.y += wave(model.xz);
			#endif
		}
	#else
		vec3 color = vaColor.rgb;
	#endif

	#ifdef ENTITY_COLOR
		color.rgb = mix(color.rgb, entityColor.rgb, entityColor.a);
	#endif

	immut vec4 model_4 = vec4(model, 1.0);
	immut vec4 view = modelViewMatrix * model_4;
	vec3 s_clip = (shadowProjection * (shadowModelView * model_4)).xyz;

	immut vec3 normal = normalize(normalMatrix * 
		#ifdef NO_NORMAL
			vec3(0.0, -1.0, 0.0)
		#else
			vaNormal
		#endif
	);

	#ifndef NO_NORMAL
		immut vec3 tangent = normalize(normalMatrix * at_tangent.xyz);

		f.tbn = mat3(tangent, cross(tangent, normal) * at_tangent.w, normal);
	#endif

	vec3 light = indexed_block_light((gbufferModelViewInverse * view).xyz, mat3(gbufferModelViewInverse) * normal);

	#ifndef NETHER
		immut float lambertian = dot(normal, shadowLightDirection);

		s_clip = distort(s_clip);
		float s_bias = max(0.00005, 0.0003 * length(s_clip.xy)) * lowp_rcp(lambertian);

		#ifdef TERRAIN
			#if SSS // todo!() check that this makes sense
				if (mc_Entity.x == 0.0 && lambertian < 0.0) s_bias *= -1.0;
			#endif
		#endif

		s_clip.z -= s_bias;
		f.s_ndc = s_clip * 0.5 + 0.5;
	#endif

	#ifdef TERRAIN
		immut float emission = clamp((mc_Entity.x + at_midBlock.w) / 15.0, 0.0, 1.0);
		light = max(light, vec3(emission)) + emission * 0.3;

		immut vec2 atlas = vec2(atlasSize);

		#if !(SM && defined MC_SPECULAR_MAP)
			f.avg_luma = srgb_luma(textureLod(gtexture, mc_midTexCoord, 4.0).rgb);
		#endif
	#else
		immut vec2 atlas = textureSize(gtexture, 0);
	#endif

	color.rgb = linear(color.rgb) * light;

	f.coord = (textureMatrix * vec4(coord, 0.0, 1.0)).xy;
	f.color = color;
	f.view = view.xyz;

	#if !(defined NM && defined MC_NORMAL_MAP)
		immut uvec2 texels = uvec2(fma(abs(f.coord - mc_midTexCoord), atlas * 2.0, vec2(0.5)));
		f.texels = bitfieldInsert(texels.x, texels.y, 16, 16);
		f.mid_coord = mc_midTexCoord;
	#endif

	gl_Position = projectionMatrix * view;
}