#include "/lib/core.glsl"
#include "/lib/config.glsl"

/* RENDERTARGETS: 1 */
layout(location = 0) out vec4 outColor;

#ifdef ALPHA_CHECK
	layout(depth_greater) out float gl_FragDepth;

	uniform float alphaTestRef;
#else
	layout(depth_unchanged) out float gl_FragDepth;
#endif

uniform float frameTimeCounter;
uniform mat4 gbufferModelViewInverse;
uniform sampler2D gtexture;

#if defined NM && defined MC_NORMAL_MAP
	uniform sampler2D normals;
#endif

#ifdef TERRAIN
	uniform ivec2 atlasSize;
#endif

in VertexData {
	vec2 coord;
	vec3 view;

	#ifndef NETHER
		vec3 s_ndc;
	#endif

	#ifndef NO_NORMAL
		flat mat3 tbn;
	#endif

	#if !(defined NM && defined MC_NORMAL_MAP)
		flat uint texels;
		flat vec2 mid_coord;
	#endif

	#ifdef TERRAIN
		vec4 color;

		#if !(SM && defined MC_SPECULAR_MAP)
			flat float avg_luma;
		#endif
	#else
		vec3 color;
	#endif
} v;

#include "/lib/fast_math.glsl"

#ifndef NETHER
	#include "/lib/skylight.glsl"
	#include "/lib/sample_shadow.glsl"
#endif

#ifdef END
	#include "/lib/rand.glsl"
#endif

#include "/lib/luminance.glsl"
#include "/lib/srgb.glsl"
#include "/lib/fog.glsl"
#include "/lib/material/specular.glsl"
#include "/lib/material/normal.glsl"

void main() {
	vec4 color = texture(gtexture, v.coord);

	#ifdef ALPHA_CHECK
		if (color.a < alphaTestRef) discard;
	#endif

	color.a = sqrt(color.a);

	#ifdef NO_NORMAL
		const mat3 tbn = mat3(vec3(0.0, 1.0, 0.0), cross(vec3(0.0, 1.0, 0.0), vec3(0.0, 0.0, 1.0)), vec3(0.0, 0.0, 1.0)); // todo!() check this
	#else
		immut mat3 tbn = v.tbn;
	#endif

	#if defined NM && defined MC_NORMAL_MAP
		immut vec3 v_tex_normal = normalize(tbn * fma(texture(normals, v.coord).rgb, vec3(2.0), vec3(-1.0)));
	#else
		#ifdef TERRAIN
			immut vec2 atlas = atlasSize;
		#else
			immut vec2 atlas = textureSize(gtexture, 0);
		#endif

		immut vec3 v_tex_normal = tbn * implicit_normal(gtexture, v.coord, v.mid_coord, v.texels, atlas, srgb_luma(color.rgb));
	#endif

	#if SM && defined MC_SPECULAR_MAP
		immut float smoothness = map_smoothness(texture(specular, v.coord).SM_CH);
	#else
		immut float avg_luma =
			#ifdef TERRAIN
				v.avg_luma;
			#else
				0.8;
			#endif

		immut float smoothness = implicit_smoothness(srgb_luma(color.rgb), avg_luma);
	#endif // this being before the sRGB -> linear conversion is intentional

	color.rgb = linear(color.rgb);

	#ifdef TERRAIN
		color *= v.color;
	#else
		color.rgb *= v.color;
	#endif

	immut vec3 n_view = normalize(v.view);
	immut float fog = fog(length(v.view));

	#ifndef NETHER
		#ifndef NO_NORMAL
			float face_lambertian = dot(v.tbn[2], shadowLightDirection);
		#endif

		vec3 lighting = AMBIENT * vec3(0.8, 0.9, 1.0);

		if (
			fog < 0.999
			#if !SSS && !defined NO_NORMAL
				&& face_lambertian > 0.01
			#endif
		) lighting += sample_shadow(
			v.view,
			n_view,
			v.s_ndc,
			skylight()

			#ifndef NO_NORMAL
				, // :concern:
				v_tex_normal,
				face_lambertian,
				smoothness
			#endif
		);

		color.rgb *= lighting;
	#endif

	#if defined END || defined NETHER
		color.rgb = mix(color.rgb, linear(fogColor), fog);
	#else
		immut float sky_fog = sky_fog(max((mat3(gbufferModelViewInverse) * n_view).y, 0.0));

		color.rgb = mix(color.rgb, sky(sky_fog, n_view), fog);
	#endif

	outColor = color;
}