#include "/lib/core.glsl"
#include "/lib/config.glsl"

#ifdef EMISSIVE_REDSTONE_BLOCK
#endif
#ifdef EMISSIVE_EMERALD_BLOCK
#endif
#ifdef EMISSIVE_LAPIS_BLOCK
#endif

uniform vec3 cameraPositionFract;
uniform mat4 modelViewMatrix, projectionMatrix, textureMatrix;

#ifndef NO_NORMAL
	uniform mat3 normalMatrix;
	in vec4 at_tangent;
#endif

#ifdef TERRAIN
	uniform ivec2 atlasSize;
	uniform vec3 chunkOffset;

	in vec2 mc_Entity;
	in vec4 at_midBlock;

	#if WAVES && defined SOLID_TERRAIN
		#include "/lib/waves/offset.glsl"
	#endif

	#if INDEXED_BLOCK_LIGHT
		uniform bool rebuildIndex;
		uniform layout(r32ui) restrict writeonly uimage1D lightIndex;

		layout(shared, binding = 1) restrict buffer indexWrite { uint len; } index_write;
	#endif
#endif

#if !defined TERRAIN || INDEXED_BLOCK_LIGHT || !(SM && defined MC_SPECULAR_MAP)
	uniform sampler2D gtexture;
#endif

#ifdef ENTITY_COLOR
	uniform vec4 entityColor;
#endif

#ifndef NO_NORMAL
	in vec3 vaNormal;
#endif

in ivec2 vaUV2;
in vec2 vaUV0;

#if defined TERRAIN || !(defined NM && defined MC_NORMAL_MAP)
	in vec2 mc_midTexCoord;
#endif

in vec3 vaPosition;
in vec4 vaColor;

out VertexData {
	vec2 coord, light;
	vec3 color;

	#ifndef NO_NORMAL
		flat mat3 tbn;
	#endif

	#if !(defined NM && defined MC_NORMAL_MAP)
		flat uint texels;
		flat vec2 mid_coord;
	#endif

	#ifdef TERRAIN
		flat uint emission;

		#if !(SM && defined MC_SPECULAR_MAP)
			flat float avg_luma;
		#endif
	#endif
} f;

#include "/lib/luminance.glsl"
#include "/lib/srgb.glsl"

void main() {
	vec3 model = vaPosition;

	#ifdef TERRAIN
		model += chunkOffset;

		#if WAVES && defined SOLID_TERRAIN
			if (mc_Entity.y > 0.5) model.y += wave(model.xz + cameraTopDown);
		#endif
	#endif

	immut vec4 view = modelViewMatrix * vec4(model, 1.0);
	immut float view_len = length(view.xyz);

	//if (view_len < far) {
		vec3 color = vaColor.rgb;

		#ifdef ENTITY_COLOR
			color.rgb = mix(color.rgb, entityColor.rgb, entityColor.a);
		#endif

		vec2 light = ((
			mat4(0.00390625, 0.0, 0.0, 0.0, 0.0, 0.00390625, 0.0, 0.0, 0.0, 0.0, 0.00390625, 0.0, 0.03125, 0.03125, 0.03125, 1.0) *
			vec4(vaUV2, 0.0, 1.0)
		).xy - 1.0/32.0) * 16.0/15.0; // lm_coord scaled to 0-1

		#ifndef NO_NORMAL
			immut vec3 normal = normalMatrix * vaNormal;
			immut vec3 tangent = normalMatrix * at_tangent.xyz;

			f.tbn = mat3(normalize(tangent), normalize(cross(tangent, normal) * normalize(at_tangent.w)), normal);
		#endif

		#ifdef TERRAIN
			#if !(SM && defined MC_SPECULAR_MAP)
				immut vec3 avg_col = textureLod(gtexture, mc_midTexCoord, 4.0).rgb;
				f.avg_luma = srgb_luma(avg_col);
			#endif

			immut float emission = min((max(mc_Entity.x, 0.0) + at_midBlock.w) / 15.0, 1.0);
			light.x = min(fma(emission, 0.3, max(light.x, emission)), 1.0);

			#if INDEXED_BLOCK_LIGHT
				if (
					rebuildIndex &&
					emission >= MIN_INDEX_LL / 15.0 && // cull too weak or non-lights
					view_len < INDEX_DIST && // V-cull outside distance
					view.z < emission && // cull behind camera
					gl_VertexID % int(exp2(2.0 + round(LOD_FALLOFF * (LAVA_LOD_BIAS * mc_Entity.y + view_len / float(INDEX_DIST))))) == 0
					// gl_VertexID % 4 == 0 once every face, so start with that and then increase times two each lod
				) {
					#if SM && defined MC_SPECULAR_MAP
						immut vec3 avg_col = textureLod(gtexture, mc_midTexCoord, 4.0).rgb;
					#endif
					
					immut uint i = atomicAdd(index_write.len, 1u);

					immut uvec3 pos = uvec3(clamp(fma(at_midBlock.xyz, vec3(1.0/64.0), 256.0 + model + cameraPositionFract), 0.0, 511.5)); // idk how i feel about this but it works
					immut uint data_r = bitfieldInsert(bitfieldInsert(bitfieldInsert(pos.x, pos.y, 9, 9), pos.z, 18, 9), uint(fma(emission, 15.0, 0.5)), 27, 4);

					// For some strange reason, multiplying with vaColor.rgb here causes flickering :/ ???
					immut uvec3 col = uvec3(fma(linear(avg_col), vec3(2047.0, 2047.0, 1023.0), vec3(0.5))); // maybe apply separateAo after this
					immut uint data_g = bitfieldInsert(bitfieldInsert(col.b, col.g, 11, 11), col.r, 22, 10);

					imageStore(lightIndex, int(i), uvec4(data_r, data_g, 0u, 0u));
				}
			#endif

			f.emission = uint(fma(emission, 15.0, 0.5));

			immut vec2 atlas = vec2(atlasSize);
		#else
			immut vec2 atlas = textureSize(gtexture, 0);
		#endif

		f.coord = (textureMatrix * vec4(vaUV0, 0.0, 1.0)).xy;
		f.color = color;
		f.light = fma(light, vec2(8191.0), vec2(0.5));
		

		#if !(defined NM && defined MC_NORMAL_MAP)
			immut uvec2 texels = uvec2(fma(abs(f.coord - mc_midTexCoord), atlas * 2.0, vec2(0.5)));
			f.texels = bitfieldInsert(texels.x, texels.y, 16, 16);
			f.mid_coord = mc_midTexCoord;
		#endif

		gl_Position = projectionMatrix * view;
	//} else gl_Position = vec4(0.0, 0.0, 8192.0, 1.0);
}