#include "/lib/core.glsl"
#include "/lib/config.glsl"

#include "/lib/tile.glsl"
layout(local_size_x = TILE, local_size_y = TILE, local_size_z = 1u) in;

uniform float frameTimeCounter;
uniform sampler2D colortex1;
uniform layout(rgba16) restrict writeonly image2D colorimg0;

#if defined COMPASS || BLOOM
	uniform vec2 pixSize;

	#include "/lib/luminance.glsl"
#endif

#ifdef COMPASS
	uniform vec3 playerLookVector;

	#include "/lib/pi.glsl"
#endif

#ifdef BLOOM
	#include "/lib/rand.glsl"
#endif

#include "/lib/tonemap.glsl"

#if DCG_EXP || DCG_COLOR || DCG_BP
	layout(shared, binding = 0) restrict readonly buffer dcgBuffer {
		vec3 max_color;
		float max_luma, min_val;
	} dcg;
#endif

void main() {
	immut ivec2 texel = ivec2(gl_GlobalInvocationID.xy);
	vec3 color = texelFetch(colortex1, texel, 0).rgb;

	#if defined COMPASS || BLOOM
		immut vec2 coord = (gl_GlobalInvocationID.xy + 0.5) * pixSize;

		// Fast math debug:
		// if (coord.y * 256.0 < lowp_rcp(coord.x)) color.g += 4.0;
		// if (coord.y * 256.0 < mediump_rcp(coord.x)) color.g += 4.0;
		// if (coord.y * 256.0 < 1.0 / coord.x) color.g -= 4.0;

		// if (coord.y < lowp_sqrt(coord.x)) color.g += 4.0;
		// if (coord.y < sqrt(coord.x)) color.g -= 4.0;
	#endif

	#if BLOOM || SATURATION != 100
		immut float luma = luminance(color);
	#endif
	
	#if BLOOM
		const vec2[] offsets = vec2[](
			vec2(0.0, 1.0),
			vec2(0.7, 0.7),
			vec2(1.0, 0.0),
			vec2(0.7,-0.7),
			vec2(0.0,-1.0),
			vec2(-0.7,-0.7),
			vec2(-1.0, 0.0),
			vec2(-0.7, 0.7)
		);

		immut float noise = rand(gl_GlobalInvocationID.xy + frameTimeCounter);
		immut float noise_cos = cos(noise);
		immut float noise_sin = sin(noise);

		immut mat2 scale_rot = 0.01 * BLOOM_RADIUS * mat2(
			noise_cos, -noise_sin,
			noise_sin, noise_cos
		);

		vec3 bloom = vec3(0.0);

		for (uint i = 0u; i < offsets.length(); ++i) {
			bloom += textureLod(colortex1, scale_rot * offsets[i] + coord, 0.0).rgb;
		}

		bloom /= offsets.length();
		bloom *= 0.1 * BLOOM * bloom * bloom * bloom;

		//color = mix(color, color * normalize(mix(bloom, luminance(bloom).xxx, 0.25)), 0.75); // crazy VHS filter thingy
		color += bloom * luma / luminance(bloom);
	#endif

	#if DCG_EXP || DCG_COLOR || DCG_BP
		color = max(color - dcg.min_val, 0.0) / dcg.max_color;
	#endif

	#if RED_MUL != 100 || GREEN_MUL != 100 || BLUE_MUL != 100
		const vec3 colorize = normalize(vec3(RED_MUL, GREEN_MUL, BLUE_MUL));

		color *= colorize / luminance(colorize);
	#endif

	#if SATURATION != 100
		color = mix(luma.rrr, color, SATURATION * 0.01);
	#endif

	#ifdef COMPASS
		const vec2 comp_pos = vec2(0.5, 0.9);
		const vec2 comp_size = vec2(0.1, 0.01);
		const float comp_line = 0.01;

		immut vec2 comp_dist = (coord - comp_pos) / comp_size;
		immut vec2 abs_dist = abs(comp_dist);

		if (abs_dist.x < 1.0 && abs_dist.y < 1.0) {
			const float inv_comp_line = 1.0 - comp_line;

			immut float ang = PI * -0.5 * comp_dist.x;
			immut float s = sin(ang);
			immut float c = cos(ang);
			immut vec2 dir = mat2(c, -s, s, c) * normalize(playerLookVector.xz);

			vec3 comp_color = vec3(0.0);

			/*
				W - < x > + E
				N - < z > + S
			*/

			comp_color.r += max(dot(dir, vec2(0.0, -1.0)) - inv_comp_line, 0.0);
			comp_color.rg += max(dot(dir, vec2(1.0, 0.0)) - inv_comp_line, 0.0);
			comp_color.rb += max(dot(dir, vec2(-1.0, 0.0)) - inv_comp_line, 0.0);
			comp_color.gb += max(dot(dir, vec2(0.0, 1.0)) - inv_comp_line, 0.0);

			comp_color = fma(comp_color, (1.0 / comp_line).xxx, vec3(max(0.1 - abs_dist.y, 0.0) * 10.0));

			color = mix(color, comp_color, luminance(comp_color) * smoothstep(0.0, 0.1, 1.5 - abs_dist.x - abs_dist.y));
		}
	#endif

	color = tonemap(color);

	imageStore(colorimg0, texel, vec4(color, luminance(color)));
}