#define GBUFFERS_BLOCK

//Settings//
#include "/lib/common.glsl"

#ifdef FSH

//Varyings//
in vec2 texCoord;
in vec2 lmCoord;
in vec3 normal, binormal, tangent;
in vec3 eastVec, northVec, sunVec, upVec;
in vec4 color;

//Uniforms//
uniform int isEyeInWater;
uniform int frameCounter;
uniform int blockEntityId;

uniform float viewWidth, viewHeight;
uniform float frameTimeCounter;

#ifdef OVERWORLD
uniform float timeBrightness, timeAngle;
uniform float shadowFade;
uniform float wetness;
#endif

uniform ivec2 eyeBrightnessSmooth;

uniform vec3 cameraPosition;
uniform vec3 skyColor;

uniform sampler2D texture;
uniform sampler2D noisetex;
uniform sampler2D gaux1;

uniform sampler3D floodfillSampler;
uniform usampler3D voxelSampler;

uniform mat4 gbufferProjectionInverse;
uniform mat4 gbufferModelViewInverse;
uniform mat4 shadowProjection;
uniform mat4 shadowModelView;

//Common Variables//
#ifdef OVERWORLD
float sunVisibility = clamp(dot(sunVec, upVec) + 0.025, 0.0, 0.1) * 10.0;
vec3 lightVec = sunVec * ((timeAngle < 0.5325 || timeAngle > 0.9675) ? 1.0 : -1.0);
#else
vec3 lightVec = sunVec;
#endif

const vec3[8] endPortalColors = vec3[8](
    vec3(0.35, 0.60, 0.75) * 1.50,
    vec3(0.60, 0.70, 1.00) * 1.40,
    vec3(0.45, 0.80, 0.90) * 1.30,
    vec3(0.35, 1.00, 1.85) * 1.20,
    vec3(0.75, 0.85, 0.65) * 1.10,
    vec3(0.40, 0.55, 0.80) * 1.00,
    vec3(0.50, 0.65, 1.00) * 0.90,
    vec3(0.55, 0.55, 0.80) * 0.80
);

//Includes//
#include "/lib/util/encode.glsl"
#include "/lib/util/ToNDC.glsl"
#include "/lib/util/ToWorld.glsl"
#include "/lib/util/ToShadow.glsl"
#include "/lib/util/bayerDithering.glsl"
#include "/lib/color/lightColor.glsl"
#include "/lib/color/netherColor.glsl"
#include "/lib/vx/voxelization.glsl"
#include "/lib/lighting/shadows.glsl"
#include "/lib/lighting/gbuffersLighting.glsl"

//Program//
void main() {
	vec4 albedo = texture2D(texture, texCoord) * color;

    if (albedo.a < 0.001) discard;

	vec3 newNormal = normal;
	float emission = 0.0;

	vec3 screenPos = vec3(gl_FragCoord.xy / vec2(viewWidth, viewHeight), gl_FragCoord.z);
	vec3 viewPos = ToNDC(screenPos);
	vec3 worldPos = ToWorld(viewPos);
	vec2 lightmap = clamp(lmCoord, 0.0, 1.0);

	float NoU = clamp(dot(newNormal, upVec), -1.0, 1.0);
	float NoL = clamp(dot(newNormal, lightVec), 0.0, 1.0);
	float NoE = clamp(dot(newNormal, eastVec), -1.0, 1.0);

	vec3 shadow = vec3(0.0);

    if (blockEntityId == 10303) {
        if (albedo.b < 10.1) {
            albedo.rgb = vec3(0.45, 0.5, 2.0) * 0.05;

            float dither = Bayer8(gl_FragCoord.xy);

            #ifdef TAA
            dither = fract(dither + frameTimeCounter * 16.0);
            #endif

            for (int j = 0; j < 4; j++) {
                float increment = (j + dither) * 0.015;
                int samples = 9;
                if (j > 2) samples = 6;
                for (int i = 1; i < samples; i++) {
                    vec3 nWorldPos = normalize(ToWorld(viewPos * i));
                         nWorldPos.xz /= nWorldPos.y;
                         nWorldPos.xz *= 0.05 * sign(-worldPos.y);
                         nWorldPos.xz *= abs(worldPos.y) + i + increment;
                         nWorldPos.xz -= vec2(cameraPosition.x, 0.0) * 0.05;

                    float particleMovement = fract((frameTimeCounter + 128.0) * (i + 8) * pow2(16.0 - i) * 0.00001);
                    vec2 coord = vec2(nWorldPos.x, nWorldPos.z + particleMovement);
                    if ((i - 1) % 2 == 0) coord = coord.yx + vec2(-1.0, 1.0) * particleMovement;
                    
                    vec3 colorSample = texture2D(texture, coord).rgb * endPortalColors[i - 1];
                         colorSample *= 1.0 / (16.0 + i);
                         colorSample *= length(colorSample) * (128.0 - float(NoU > 0.95) * 64.0);
                    albedo.rgb += colorSample * clamp(1.0 - length(nWorldPos.xz) * 0.1, 0.0, 1.0);
                }
            }
        }
    } else {
        gbuffersLighting(albedo, screenPos, viewPos, worldPos, shadow, lightmap, NoU, NoL, NoE, 0.0, 0.0, emission);
    }

	/* DRAWBUFFERS:03 */
	gl_FragData[0] = albedo;
	gl_FragData[1] = vec4(encodeNormal(newNormal), emission * 0.1, 1.0);
}

#endif

/////////////////////////////////////////////////////////////////////////////////////

#ifdef VSH

//Varyings//
out vec2 texCoord;
out vec2 lmCoord;
out vec3 normal, binormal, tangent;
out vec3 eastVec, northVec, sunVec, upVec;
out vec4 color;

//Uniforms//
#ifdef TAA
uniform int framemod8;

uniform float viewWidth, viewHeight;
#endif

#if defined OVERWORLD || defined END
uniform float timeAngle;
#endif

uniform mat4 gbufferModelView, gbufferModelViewInverse;

//Attributes//
attribute vec4 at_tangent;

//Includes
#ifdef TAA
#include "/lib/util/jitter.glsl"
#endif

//Program//
void main() {
	//Coord
	texCoord = (gl_TextureMatrix[0] * gl_MultiTexCoord0).xy;

	//Lightmap Coord
	lmCoord = (gl_TextureMatrix[1] * gl_MultiTexCoord1).xy;
	lmCoord = clamp((lmCoord - 0.03125) * 1.06667, vec2(0.0), vec2(0.9333, 1.0));

	//Normal, Binormal and Tangent
	normal = normalize(gl_NormalMatrix * gl_Normal);
	binormal = normalize(gl_NormalMatrix * cross(at_tangent.xyz, gl_Normal.xyz) * at_tangent.w);
	tangent = normalize(gl_NormalMatrix * at_tangent.xyz);

	//Sun & Other vectors
	#if defined OVERWORLD || defined END
	sunVec = getSunVector(gbufferModelView, timeAngle);
	#endif
	
	upVec = normalize(gbufferModelView[1].xyz);
	northVec = normalize(gbufferModelView[2].xyz);
	eastVec = normalize(gbufferModelView[0].xyz);

	//Color & Position
	vec4 position = gbufferModelViewInverse * gl_ModelViewMatrix * gl_Vertex;

	color = gl_Color;
	if (color.a < 0.1) color.a = 1.0;

	gl_Position = gl_ProjectionMatrix * gbufferModelView * position;

	#ifdef TAA
	gl_Position.xy = TAAJitter(gl_Position.xy, gl_Position.w);
	#endif
}

#endif