#include "/lib/core.glsl"
#include "/lib/config.glsl"

out gl_PerVertex {
	vec4 gl_Position;
};

uniform float far;
uniform vec3 shadowLightDirection, shadowProjScale;
uniform mat3 normalMatrix;
uniform mat4 gbufferModelViewInverse, modelViewMatrix, projectionMatrix, shadowModelView, textureMatrix;

#ifndef NO_NORMAL
	#include "/lib/tbn/vsh.glsl"
#endif

#if defined TERRAIN || !(defined NM && defined MC_NORMAL_MAP)
	in vec2 mc_midTexCoord;
#endif

#ifdef TERRAIN
	uniform ivec2 atlasSize;
	uniform vec3 chunkOffset;

	in vec2 mc_Entity;
	in vec4 at_midBlock;

	#if WAVES
		#include "/lib/waves/offset.glsl"
	#endif
#endif

#ifdef ENTITY_COLOR
	uniform vec4 entityColor;
#endif

#if !defined TERRAIN || !(SM && defined MC_SPECULAR_MAP)
	uniform sampler2D gtexture;
#endif

in vec2 vaUV0;
in vec3 vaPosition;
in vec4 vaColor;

out VertexData {
	layout(location = 2) vec3 tint;
	layout(location = 2, component = 3) float ao;
	layout(location = 3) vec3 light;
	layout(location = 4) vec3 view;
	layout(location = 5) vec2 coord;

	#ifdef TERRAIN
		layout(location = 1, component = 3) flat float alpha;

		#if !(SM && defined MC_SPECULAR_MAP)
			layout(location = 6) flat float avg_luma;
		#endif
	#endif

	#ifndef NETHER
		layout(location = 7) vec3 s_screen;
	#endif

	#if !(defined NM && defined MC_NORMAL_MAP)
		layout(location = 8) flat uvec2 uv_data;
	#endif
} v;

#include "/lib/srgb.glsl"
#include "/lib/shadow_bias.glsl"
#include "/lib/vertex_block_light.glsl"
#include "/lib/distort.glsl"

void main() {
	v.ao = vaColor.a;
	vec3 model = vaPosition;

	#ifdef TERRAIN
		model += chunkOffset;

		if (mc_Entity.y > 0.5) {
			v.alpha = WATER_OPACITY * 0.01;

			#if WAVES
				model.y += wave(f16vec2(model.xz));
			#endif
		} else v.alpha = 1.0;
	#endif

	if (length(model.xz) < far) {
		immut vec3 view = mat4x3(modelViewMatrix) * vec4(model, 1.0);
		gl_Position = projectionMatrix * vec4(view, 1.0);
		v.view = view.xyz;

		immut vec3 normal = normalize(normalMatrix * 
			#ifdef NO_NORMAL
				vec3(0.0, -1.0, 0.0)
			#else
				vaNormal
			#endif
		);

		#ifndef NO_NORMAL
			tbn_comp.normal = normal;
			tbn_comp.tangent = normalMatrix * at_tangent.xyz;
			tbn_comp.handedness = at_tangent.w;
		#endif

		f16vec3 light = indexed_block_light(mat3(gbufferModelViewInverse) * view, f16vec3(mat3(gbufferModelViewInverse) * normal));

		#ifndef NETHER
			immut float16_t lambertian = float16_t(dot(normal, shadowLightDirection));
			vec2 bias = shadow_bias(lambertian);

			#if SSS && defined TERRAIN
				// todo!() check that this makes sense
				if (mc_Entity.x == 0.0 && lambertian < float16_t(0.0)) bias *= -1.0;
			#endif

			vec3 s_ndc = shadowProjScale * (mat3(shadowModelView) * (
				40.0 / shadowMapResolution * bias.y * normal + model
			));
			s_ndc.z -= bias.x / shadowMapResolution;

			v.s_screen = fma(vec3(distort(s_ndc.xy), s_ndc.z), vec3(0.5), vec3(0.5));
		#endif

		#ifdef TERRAIN
			immut float16_t emission = min((max(float16_t(mc_Entity.x), float16_t(0.0)) + float16_t(at_midBlock.w)) / float16_t(15.0), float16_t(1.0));
			light.x = min(fma(emission, float16_t(0.3), max(light.x, emission)), float16_t(1.0));

			#if !(SM && defined MC_SPECULAR_MAP)
				v.avg_luma = luminance(vaColor.rgb * textureLod(gtexture, mc_midTexCoord, 4.0).rgb);
			#endif

			immut vec2 atlas = vec2(atlasSize * 2);
		#else
			immut vec2 atlas = vec2(textureSize(gtexture, 0) * 2);
		#endif

		vec3 color = vaColor.rgb;
		#ifdef ENTITY_COLOR
			color.rgb = mix(color.rgb, entityColor.rgb, entityColor.a);
		#endif
		v.tint = linear(color);
		v.light = light;
		v.coord = mat4x2(textureMatrix) * vec4(vaUV0, 0.0, 1.0);

		#if !(defined NM && defined MC_NORMAL_MAP)
			immut uvec2 texels = uvec2(fma(abs(v.coord - mc_midTexCoord), atlas, vec2(0.5)));
			v.uv_data = uvec2(packUnorm2x16(mc_midTexCoord), bitfieldInsert(texels.x, texels.y, 16, 16));
		#endif
	} else gl_Position = vec4(0.0, 0.0, 1.0/0.0, 1.0);
}