#include "/lib/core.glsl"
#include "/lib/config.glsl"

/* RENDERTARGETS: 1 */
layout(location = 0) out vec4 colortex1;

#ifdef ALPHA_CHECK
	layout(depth_greater) out float gl_FragDepth;

	uniform float alphaTestRef;
#else
	layout(depth_unchanged) out float gl_FragDepth;
#endif

uniform mat4 gbufferModelViewInverse;
uniform sampler2D gtexture;

#ifdef TERRAIN
	uniform ivec2 atlasSize;
#endif

#ifndef NO_NORMAL
	#include "/lib/tbn/fsh.glsl"
#endif

in VertexData {
	layout(location = 2) vec3 tint;
	layout(location = 2, component = 3) float ao;
	layout(location = 3) vec3 light;
	layout(location = 4) vec3 view;
	layout(location = 5) vec2 coord;

	#ifdef TERRAIN
		layout(location = 1, component = 3) flat float alpha;

		#if !(SM && defined MC_SPECULAR_MAP)
			layout(location = 6) flat float avg_luma;
		#endif
	#endif

	#ifndef NETHER
		layout(location = 7) vec3 s_screen;
	#endif

	#if !(defined NM && defined MC_NORMAL_MAP)
		layout(location = 8) flat uvec2 uv_data;
	#endif
} v;

#ifndef NETHER
	uniform vec3 shadowLightDirection;
	
	#include "/lib/skylight.glsl"
	#include "/lib/distort.glsl"
	#include "/lib/sample_shadow.glsl"
#endif

#ifdef END
	uniform float frameTimeCounter;
	#include "/lib/rand.glsl"
#endif

#include "/lib/luminance.glsl"
#include "/lib/srgb.glsl"
#include "/lib/fog.glsl"
#include "/lib/material/specular.glsl"
#include "/lib/material/normal.glsl"

void main() {
	f16vec4 color = f16vec4(texture(gtexture, v.coord));

	#ifdef ALPHA_CHECK
		if (color.a < float16_t(alphaTestRef)) discard;
	#endif

	immut f16vec3 tint = f16vec3(v.tint);

	#ifdef TERRAIN
		color *= f16vec4(tint, v.alpha);
	#else
		color.rgb *= tint;
	#endif

	#ifdef NO_NORMAL
		immut vec3 normal = normalize(mat3(gbufferModelViewInverse) * vec3(0.0, 0.0, 1.0));
		immut vec3 tangent = normalize(mat3(gbufferModelViewInverse) * vec3(0.0, 1.0, 0.0));
		immut mat3 tbn = mat3(tangent, normalize(cross(tangent, normal)), normal);
	#else
		immut mat3 tbn = get_tbn();
	#endif

	#if SM && defined MC_SPECULAR_MAP
		immut float16_t smoothness = map_smoothness(float16_t(texture(specular, v.coord).SM_CH));
	#else
		immut float16_t avg_luma =
			#ifdef TERRAIN
				float16_t(v.avg_luma);
			#else
				float16_t(0.8);
			#endif

		immut float16_t smoothness = gen_smoothness(luminance(color.rgb), avg_luma);
	#endif

	#if defined NM && defined MC_NORMAL_MAP
		immut f16vec3 v_tex_normal = normalize(f16vec3(tbn * fma(texture(normals, v.coord).rgb, vec3(2.0), vec3(-1.0))));
	#else
		#ifdef TERRAIN
			immut vec2 atlas = vec2(atlasSize);
		#else
			immut vec2 atlas = vec2(textureSize(gtexture, 0));
		#endif

		immut f16vec3 v_tex_normal = normalize(f16vec3(tbn * gen_normal(gtexture, tint, v.coord, v.uv_data, atlas, luminance(color.rgb))));
	#endif

	color.rgb = linear(color.rgb);

	f16vec3 lighting = f16vec3(v.light);

	#ifndef NETHER
		immut f16vec3 skylight = skylight();

		#ifdef NO_NORMAL
			const float16_t face_lambertian = float16_t(1.0);
		#else
			immut float16_t face_lambertian = float16_t(dot(tbn[2], shadowLightDirection));
		#endif

		#if SSS
			f16vec3 light = sample_shadow(v.s_screen);
		#endif

		if (face_lambertian > float16_t(0.01)) {
			immut float16_t roughness = float16_t(1.0) - sqrt(smoothness);

			#if !SSS
				f16vec3 light = sample_shadow(v.s_screen);
			#endif

			if (dot(light, f16vec3(1.0)) > float16_t(0.0)) {
				immut f16vec3 shadow_light_dir = f16vec3(shadowLightDirection);
				immut float16_t tex_lambertian = dot(v_tex_normal, shadow_light_dir);

				immut float16_t specular = brdf(v_tex_normal, f16vec3(normalize(v.view)), shadow_light_dir, roughness) * float16_t(0.03);
				immut float16_t diffuse = tex_lambertian * (float16_t(1.0) - specular) / PI_16;

				light *= float16_t(3.0) * (diffuse + specular / max(color.rgb, float16_t(1.0e-5)));

				lighting += light * skylight;
			}
		}

		#if SSS
			else {
				lighting += float16_t(0.25) * skylight * light;
			}
		#endif
	#endif

	color.rgb *= lighting;

	immut vec3 pe = mat3(gbufferModelViewInverse) * v.view;
	immut float16_t fog = fog(pe);

	#if defined END || defined NETHER
		color.rgb = mix(color.rgb, linear(f16vec3(fogColor)), fog);
	#else
		immut vec3 n_pe = normalize(pe);
		immut float16_t sky_fog = sky_fog(float16_t(n_pe.y));
		color.rgb = mix(color.rgb, f16vec3(sky(sky_fog, n_pe, mat3(gbufferModelViewInverse) * shadowLightDirection)), fog);
	#endif

	colortex1 = color;
}