#include "/lib/core.glsl"
#include "/lib/config.glsl"

#ifdef EMISSIVE_REDSTONE_BLOCK
#endif
#ifdef EMISSIVE_EMERALD_BLOCK
#endif
#ifdef EMISSIVE_LAPIS_BLOCK
#endif

out gl_PerVertex {
	vec4 gl_Position;
};

uniform float far;
uniform vec3 cameraPositionFract;
uniform mat4 gbufferModelViewInverse, modelViewMatrix, projectionMatrix, textureMatrix;

#ifdef TERRAIN
	uniform ivec2 atlasSize;
	uniform vec3 cameraPosition, chunkOffset;

	in vec2 mc_Entity;
	in vec4 at_midBlock;

	#if WAVES && defined SOLID_TERRAIN
		#include "/lib/waves/offset.glsl"
	#endif

	#if INDEXED_BLOCK_LIGHT && !defined LIGHT_LEVELS
		uniform bool rebuildIndex;
		#include "/buf/index.glsl"
	#endif

	#include "/lib/rand.glsl"
#endif

#ifdef ENTITY_COLOR
	uniform vec4 entityColor;
#endif

#if !defined TERRAIN || (INDEXED_BLOCK_LIGHT && !defined LIGHT_LEVELS) || !(SM && defined MC_SPECULAR_MAP)
	uniform sampler2D gtexture;
#endif

#ifndef NO_NORMAL
	uniform mat3 normalMatrix;
	#include "/lib/tbn/vsh.glsl"
#endif

in vec2 vaUV0;

#if defined TERRAIN || !(defined NM && defined MC_NORMAL_MAP)
	in vec2 mc_midTexCoord;
#endif

in vec3 vaPosition;
in vec4 vaColor;

out VertexData {
	layout(location = 2) vec2 coord;
	layout(location = 2, component = 2) vec2 light;
	layout(location = 3) vec3 tint;
	layout(location = 3, component = 3) float ao;

	#if !(defined NM && defined MC_NORMAL_MAP)
		layout(location = 4) flat uvec2 uv_data;
	#endif

	#ifdef TERRAIN
		layout(location = 4, component = 2) flat uint emission;

		#if !(SM && defined MC_SPECULAR_MAP)
			layout(location = 1, component = 3) flat float avg_luma;
		#endif
	#endif
} v;

#include "/lib/luminance.glsl"
#include "/lib/srgb.glsl"
#include "/lib/norm_uv2.glsl"

void main() {
	v.ao = vaColor.a;
	vec3 model = vaPosition;

	#ifdef TERRAIN
		model += chunkOffset;

		#if WAVES && defined SOLID_TERRAIN
			if (mc_Entity.y > 0.5) model.y += wave(f16vec2(model.xz) + f16vec2(cameraTopDown));
		#endif
	#endif

	if (length(model.xz) < far) {
		immut vec3 view = mat4x3(modelViewMatrix) * vec4(model, 1.0);
		immut vec4 clip = projectionMatrix * vec4(view, 1.0);
		gl_Position = clip;

		f16vec3 color = f16vec3(vaColor.rgb);
		#ifdef ENTITY_COLOR
			color.rgb = mix(color.rgb, f16vec3(entityColor.rgb), float16_t(entityColor.a));
		#endif
		v.tint = color;

		f16vec2 light = norm_uv2();

		#ifdef TERRAIN
			#if !(SM && defined MC_SPECULAR_MAP)
				immut f16vec3 avg_col = color * f16vec3(textureLod(gtexture, mc_midTexCoord, 4.0).rgb);
				v.avg_luma = luminance(avg_col);
			#endif

			float16_t emission = min((max(float16_t(mc_Entity.x), float16_t(0.0)) + float16_t(at_midBlock.w)) / float16_t(15.0), float16_t(1.0));
			light.x = min(fma(float16_t(emission), float16_t(0.3), max(light.x, emission)), float16_t(1.0));

			emission *= float16_t(15.0);
			v.emission = uint(emission + float16_t(0.5));

			#if INDEXED_BLOCK_LIGHT && !defined LIGHT_LEVELS
				immut f16vec3 view_16 = f16vec3(view);
				immut float16_t dist = length(view_16);
				immut f16vec3 abs_model = abs(f16vec3(model));

				if (
					// only rebuild the index once every INDEX_RATE frames
					rebuildIndex &&
					// run once per face
					gl_VertexID % 4 == 1 &&
					// cull too weak or non-lights
					emission >= float16_t(MIN_INDEX_LL) &&
					// cull outside INDEX_DIST using Chebyshev distance
					max3(abs_model.x, abs_model.y, abs_model.z) < float16_t(INDEX_DIST) &&
					// cull behind camera outside of illumination range
					(view_16.z < float16_t(0.0) || dist <= emission)
				) {
					immut bool fluid = mc_Entity.y > 0.0;
					immut vec3 probably_world = model + cameraPosition;

					// LOD culling
					// increase times two each LOD
					// the fact that the values resulting from higher LODs are divisible by the lower ones means that no lights will appear only further away
					if (uint8_t(rand(round(vec2(probably_world.x, probably_world.y + probably_world.z))) * 255.0) % (uint8_t(1u) << uint8_t(min(7.0, fma(
						float16_t(LAVA_LOD_BIAS) * float16_t(fluid) + dist / float16_t(INDEX_DIST),
						float16_t(LOD_FALLOFF),
						float16_t(0.5)
					)))) == uint8_t(0u)) {
						#if SM && defined MC_SPECULAR_MAP
							immut f16vec3 avg_col = f16vec3(textureLod(gtexture, mc_midTexCoord, 4.0).rgb);
						#endif

						immut uint i = atomicAdd(index.queue, 1u);

						immut uvec3 light_pe = uvec3(clamp(fma(at_midBlock.xyz, vec3(1.0/64.0), 256.0 + mat3(gbufferModelViewInverse) * view + cameraPositionFract), 0.0, 511.5)); // this feels slightly cursed but it works // somehow
						index.data[i] = bitfieldInsert(
							bitfieldInsert(
								bitfieldInsert(bitfieldInsert(light_pe.x, light_pe.y, 9, 9), light_pe.z, 18, 9), // color
								v.emission, 27, 4 // intensity
							),
							uint(fluid), 31, 1 // "wide" flag (currently set for lava)
						);

						immut uvec3 col = uvec3(fma(linear(color * avg_col), f16vec3(31.0, 63.0, 31.0), f16vec3(0.5)));
						index.color[i] = uint16_t(bitfieldInsert(bitfieldInsert(col.g, col.r, 6, 5), col.b, 11, 5));
					}
				}
			#endif

			immut vec2 atlas = vec2(2 * atlasSize);
		#else
			immut vec2 atlas = vec2(2 * textureSize(gtexture, 0));
		#endif

		v.light = light;
		v.coord = mat4x2(textureMatrix) * vec4(vaUV0, 0.0, 1.0);

		#if !(defined NM && defined MC_NORMAL_MAP)
			immut uvec2 texels = uvec2(fma(abs(v.coord - mc_midTexCoord), atlas, vec2(0.5)));
			v.uv_data = uvec2(packUnorm2x16(mc_midTexCoord), bitfieldInsert(texels.x, texels.y, 16, 16));
		#endif

		#ifndef NO_NORMAL
			init_tbn_w();
		#endif
	} else gl_Position = vec4(0.0/0.0, 0.0/0.0, 1.0/0.0, 0.0);
}