#include "/lib/core.glsl"
#include "/lib/config.glsl"

/* Color Grading, Tone Mapping and Misc. Post-Processing */

layout(local_size_x = 16, local_size_y = 16, local_size_z = 1) in;
const vec2 workGroupsRender = vec2(1.0, 1.0);

uniform sampler2D colortex1;
uniform layout(rgba16) restrict writeonly image2D colorimg0;

#ifdef COMPASS
	#include "/lib/view.glsl"
	uniform vec3 playerLookVector;

	#include "/lib/pi.glsl"
#endif

#include "/lib/fast_math.glsl"
#include "/lib/tonemap.glsl"

#if DCG_EXP || DCG_COLOR || DCG_BP
	#include "/buf/dcg.glsl"
#endif

void main() {
	immut i16vec2 texel = i16vec2(gl_GlobalInvocationID.xy);
	vec3 color = texelFetch(colortex1, texel, 0).rgb;

	#if DCG_EXP || DCG_COLOR || DCG_BP
		color = max(color - dcg.min_val, 0.0) / dcg.max_color;
	#endif

	#if RED_MUL != 100 || GREEN_MUL != 100 || BLUE_MUL != 100
		const vec3 colorize = normalize(vec3(RED_MUL, GREEN_MUL, BLUE_MUL));

		color *= colorize / luminance(colorize);
	#endif

	#if SATURATION != 100
		color = mix(luminance(color).rrr, color, SATURATION * 0.01);
	#endif

	#ifdef COMPASS
		immut vec2 coord = (gl_GlobalInvocationID.xy + 0.5) / view();

		// Fast math debug:
		// if (coord.y * 256.0 < lowp_rcp(coord.x)) color.g += 4.0;
		// if (coord.y * 256.0 < mediump_rcp(coord.x)) color.g += 4.0;
		// if (coord.y * 256.0 < 1.0 / coord.x) color.g -= 4.0;

		// if (coord.y < lowp_sqrt(coord.x)) color.g += 4.0;
		// if (coord.y < sqrt(coord.x)) color.g -= 4.0;

		const vec2 comp_pos = vec2(0.5, 0.9);
		const vec2 comp_size = vec2(0.1, 0.01);
		const float comp_line = 0.01;

		immut vec2 comp_dist = (coord - comp_pos) / comp_size;
		immut vec2 chebyshev_dist = abs(comp_dist);

		if (chebyshev_dist.x < 1.0 && chebyshev_dist.y < 1.0) {
			const float inv_comp_line = 1.0 - comp_line;

			immut float ang = PI * -0.5 * comp_dist.x;
			immut float s = sin(ang);
			immut float c = cos(ang);
			immut vec2 dir = mat2(c, -s, s, c) * normalize(playerLookVector.xz);

			vec3 comp_color = vec3(0.0);

			/*
				W - < x > + E
				N - < z > + S
			*/

			comp_color.r += max(dot(dir, vec2(0.0, -1.0)) - inv_comp_line, 0.0);
			comp_color.rg += max(dot(dir, vec2(1.0, 0.0)) - inv_comp_line, 0.0);
			comp_color.rb += max(dot(dir, vec2(-1.0, 0.0)) - inv_comp_line, 0.0);
			comp_color.gb += max(dot(dir, vec2(0.0, 1.0)) - inv_comp_line, 0.0);

			comp_color = fma(comp_color, (1.0 / comp_line).xxx, vec3(max(0.1 - chebyshev_dist.y, 0.0) * 10.0));

			color = mix(color, comp_color, luminance(comp_color) * smoothstep(0.0, 0.1, 1.5 - chebyshev_dist.x - chebyshev_dist.y));
		}
	#endif

	color = tonemap(color);

	imageStore(colorimg0, texel, vec4(color, 0.0));
}