#include "/lib/core.glsl"
#include "/lib/config.glsl"

/* Dynamic Color Grading Preparation */

const ivec3 workGroups = ivec3(1, 1, 1);

writeonly
#include "/buf/indirect.glsl"

writeonly
#include "/buf/indirect_sm_lit.glsl"

#if DCG_EXP || DCG_COLOR || DCG_BP
	layout(local_size_x = 8, local_size_y = 8, local_size_z = 1) in;

	uniform float frameTime;
	uniform sampler2D colortex1;

	// #include "/lib/view.glsl"
	// uniform layout(rgb10_a2) restrict writeonly image2D colorimg1;

	#include "/buf/dcg.glsl"
	#include "/lib/tonemap.glsl"

	shared uvec3 sh_max_color;
	shared uint sh_min_val;
#else
	layout(local_size_x = 1, local_size_y = 1, local_size_z = 1) in;
#endif

void main() {
	if (gl_LocalInvocationIndex == 0) {
		indirect.work_groups.x = 0u;
		indirect_sm_lit.work_groups.x = 0u;
	}

	#if DCG_EXP || DCG_COLOR || DCG_BP
		const uint16_t u16_max = uint16_t(0x0000FFFFu);

		if (gl_LocalInvocationIndex == 0u) {
			sh_min_val = 0x7f800000u; // f32_inf
			sh_max_color = uvec3(0u);
		}

		barrier();

		vec3 max_color = vec3(0.0);
		float min_val = DCG_MAX_BP;

		const f16vec2 coord_scale = f16vec2(1.0 / vec2(gl_WorkGroupSize.xy * DCG_SAMPLES));
		immut f16vec2 local_coord = fma(f16vec2(gl_LocalInvocationID.xy), coord_scale, float16_t(0.5) * coord_scale);

		for (uint x = 0u; x < DCG_SAMPLES; ++x) for (uint y = 0u; y < DCG_SAMPLES; ++y) {
			immut f16vec2 coord = f16vec2(vec2(x, y) / float(DCG_SAMPLES)) + local_coord;
			immut vec3 color = textureLod(colortex1, coord, 0.0).rgb;
			// imageStore(colorimg1, ivec2(coord * view()), vec4(1.0, 1.0, 1.0, 0.0));

			min_val = min(min_val, min3(color.r, color.g, color.b));
			max_color = mix(max(max_color, color), max_color, exp2(-DCG_PEAK_SENS * 0.075 / float(DCG_SAMPLES)));
		}

		atomicMin(sh_min_val, floatBitsToUint(min_val));

		immut uvec3 scaled_max = uvec3(fma(max_color, vec3(u16_max), vec3(0.5)));
		atomicAdd(sh_max_color.r, scaled_max.r);
		atomicAdd(sh_max_color.g, scaled_max.g);
		atomicAdd(sh_max_color.b, scaled_max.b);

		barrier();

		if (gl_LocalInvocationIndex == 0u) {
			immut vec3 tonemapped_min = tonemap((uintBitsToFloat(sh_min_val) * DCG_BP * 0.1).rrr);
			min_val = clamp(min3(tonemapped_min.r, tonemapped_min.g, tonemapped_min.b), 0.0, 1.0); // clamp to prevent NaNs
			max_color = max(pow(vec3(sh_max_color) / float(u16_max * gl_WorkGroupSize.x * gl_WorkGroupSize.y), vec3(0.06 * DCG_EXP)), 0.0);

			immut float max_luma = luminance(max_color);
			max_color = mix(vec3(max_luma), max_color, 0.1 * DCG_COLOR) - min_val;

			dcg.max_color = mix(max_color, dcg.max_color, exp2(-DCG_COLOR_SPEED * frameTime));
			dcg.min_val = mix(min_val, dcg.min_val, exp2(-DCG_BP_SPEED * frameTime));
			dcg.max_luma = mix(max_luma, dcg.max_luma, exp2(-DCG_EXP_SPEED * frameTime));
		}
	#endif
}