;===============================================;
; S-CPU IRQ System                              ;
;===============================================;

; This is a dedicated version of the S-CPU IRQ Controller made for ZSNES compatibility.
; During the boot, if ZSNES is detected, this version is uploaded instead of the standard
; one. The main differences are in the wait for h-blank routines, as well the priority of
; the executions. There is no way to know on ZSNES if an IRQ was generated by PPU, SA-1 or
; both. For that reason, when an IRQ is executed, immediately it's checked if it's an IRQ
; executed by SA-1 and afterwards, if an actual V-IRQ happens, it will be executed in an
; an other IRQ.

irq_wram_copy:					; Upload address
base $1D00					; Base WRAM address

snes_irq_mem:
	db $00					; $1D00: Bit 7 -- next IRQ is NMI-on-IRQ V-Blank (clears on V-Blank end).
						; $1D00: Bit 0 -- enable NMI-on-IRQ V-Blank.
	db $00					; $1D01: Which scanline to run NMI-on-IRQ execute ptr.
	db $00					; $1D02: Flag to enable Custom Timer IRQ ptr.
	db $00					; $1D03: SA-1 reg $2224 mirror.
	dl $000000				; $1D04-$1D06: Custom timer IRQ execute pointer.
	db $00					; $1D07: Reserved for future expansion.

fire_nmi_irq:					; Routine to fire IRQ-as-NMI.
	LDA.w snes_irq_mem+0			; \ IRQ-as-NMI enabled?
	BEQ .no_irq_as_nmi			;  | Set up next IRQ so it fires it.
	ORA #$80				;  |
	STA.w snes_irq_mem+0			; /
	LDA.w snes_irq_mem+1			; \ Set V-Count fire position
	STA $4209				;  |
	STZ $420A				; /
	LDA #$21				; \ Enable IRQ but no NMI.
	STA $4200				; / Keep joypad.
						;
.no_irq_as_nmi:					; \ Return
	RTS					; /
	
snes_irq:					; IRQ Start
	REP #$30				; \ Preserve A/X/Y/D/B
	PHA					;  |
	PHX					;  |
	PHY					;  |
	PHD					;  |
	PHB					;  |
	PHK					;  |
	PLB					;  |
	LDA.w #$3000				;  |
	TCD					; /
	SEP #$30				; 8-bit A/X/Y
						;
	LDA.w snes_irq_mem+3			; \ Preserve BW-RAM Mapping and
	STZ.w snes_irq_mem+3			;  | reset to default value.
	STZ $2224				;  |
	PHA					; /
						;
	LDA $2300				; \ Branch if it's an IRQ from SA-1 CPU
	BMI .sa1_irq				; /
						;
	AND #$20				; \ Otherwise, branch if it's an IRQ from
	BNE .sa1_cc1				; / Character Conversion 1.
						;
	LDA $4211				; \ If it's not a V-/H-IRQ, leave IRQ routine.
	BPL .done				; / We have no idea what we're doing here.
						;
	LDA.w snes_irq_mem+0			; \ Branch if it's an IRQ-as-NMI
	BMI .maybe_nmi				; /
	
.ppu_irq:					; Otherwise, it's a regular V-/H-IRQ.
	LDA.w snes_irq_mem+2			; \ If no custom IRQ code enable, branch
	BEQ .regular_ppu			; / to regular SMW IRQ.
	
.custom_ppu:					; \ Else, jump to the custom IRQ code pointer
	PHK					;  | and return at the end of IRQ code.
	PEA.w .done-1				;  |
	JML.w [snes_irq_mem+4]			; /
	
.regular_ppu:					; \ If the normal SMW IRQ code is used, call it.
	JSL irq_ppu_main			;  | and then return to the IRQ end code.
	BRA .done				; /

.maybe_nmi:					; \ If IRQ-as-NMI flag is not set, run regular
	ASL					;  | IRQ code instead.
	BEQ .ppu_irq				; /

.yes_nmi:					; \ Otherwise, jump to the NMI routine.
	JML snes_nmi_main			; /
	
.sa1_cc1:					; \ If it's a Character Conversion IRQ,
	STA $2202				;  | set the CC1 flag, clear CC1 IRQ register
	STA $318D				;  | and return.
	BRA .done				; /
	
.sa1_irq:					; \ If it's a SA-1 IRQ, clear the flag from
	AND #$80				;  | SA-1 IRQ register.
	STA $2202				; /
						;
	CLI					; Enable interrupts
						;
	PHK					; \ Jump to the requested SA-1 pointer
	PEA.w .return-1				;  |
	JML [$3183]				; /
	
.return:					; SA-1 IRQ return.
	SEP #$34				; Disable interrupts and make sure A/X/Y is 8-bit
	INC $318A				; Set ready flag.
	
.done:						; We're done
	PLA					; \ Restore BW-RAM Mapping
	STA $2224				;  |
	STA.w snes_irq_mem+3			; /
	
	REP #$30				; \ Restore all internal registers and
	PLB					;  | finish IRQ.
	PLD					;  |
	PLY					;  |
	PLX					;  |
	PLA					;  |
	RTI					; /
	
warnpc $001E00					; WRAM Memory Limit
base off

irq_wram_copy_end:

; SMW's PPU IRQ code goes below.

wait_for_hblank:				; Terrible as the original.
	REP #$20
	PHA
	
.wait:
	SEP #$20
	LDA $2137
	LDA $213C
	XBA
	LDA $213C
	XBA
	REP #$20
	AND #$01FF
	CMP #$0100
	BCC .wait
	
	PLA
	SEP #$20
	NOP #10
	RTS

irq_ppu_main:
	LDA #$81

	BIT $6D9B
	BMI .mode_7
	
	STA $4200

	LDA $25
	XBA
	LDA $22
	LDX $23
	LDY $24
	
	PEA $2100
	PLD
	
	JSR wait_for_hblank
		
	STA $11
	STX $11
	STY $12
	XBA
	STA $12
	LDA $303E
	STA $05
	LDA $3040
	STA $31
	
	PEA $3000
	PLD
	
	JSR.w fire_nmi_irq
	RTL
	
.mode_7:
	BVC .platform
	
	LDY $11
	BEQ .first_irq

.normal:	
	STA $4200
	
	LDX $1A
	LDY $1B
	LDA $1D
	XBA
	LDA $1C
	CLC
	ADC $7888
	
	PEA $2100
	PLD
	
	JSR wait_for_hblank
	
	STX $0D
	STY $0D
	STA $0E
	XBA
	STA $0E
	
	LDA #$59
	STA $07
	LDA #$07
	STA $0B
	
	LDA $303E
	STA $05
	LDA $3040
	STA $31
	
	PEA $3000
	PLD
	RTL

.first_irq:
	INC $11

	LDA #$AE
	SEC
	SBC $7888
	STA $4209
	STZ $420A
	
	LDA #$A1
	
.platform:
	LDY $7493
	BEQ .apply
	
	LDY $7495
	CPY #$40
	BCC .apply
	
	LDA #$81
	BRA .normal
	
.apply:
	STA $4200
	
	LDA $3A
	LDX $3B
	LDY $3C

	PEA $2100
	PLD
	
	JSR wait_for_hblank
	
	STA $0D
	STX $0D
	STY $0E
	LDA $303D
	STA $0E
	LDA #$07
	STA $05
	
	PEA $3000
	PLD
	RTL
