#include <cstdio>
#include <cctype>
#include <cstring>
#include <cstdlib>
#include <dirent.h>
#include "asar/asardll.h"

//use 16MB ROM size to avoid asar malloc/memcpy on 8MB of data per block.
#define MAX_ROM_SIZE 16*1024*1024

// Maximum block range.
#define MAX_BLOCKS 0x8000
#define MIN_BLOCKS 0x200

#define GPS_EXE_VERSION "1.4.5"
#define GPS_ASM_VERSION 3

struct display {
	char* description;
	int pswitch = -MAX_BLOCKS; // map16 tile to show when the p-switch is active
	bool pswitch_relative = false; // if the map16 tile is relative to the current one
	int hidden = -MAX_BLOCKS; // map16 tile to show when invisble objects are active
	bool hidden_relative = false; // whether the invisible object map16 is relative
	int sprite = -MAX_BLOCKS; // sprite map16 tile to show
	bool transparent = false; // display the tile as transparent
};

struct block{
	short acts_like = -1;
	char *file_name = nullptr;
	int line = 0;
	int pointer;
	display editordisplay;
	~block()
	{
		if (file_name) {
			delete []file_name;
		}
	}
};

struct block_table_data{
	unsigned char banks[MAX_BLOCKS] = {0x00};
	unsigned char pointers[MAX_BLOCKS*2] = {0x00};
	unsigned char acts_likes[MAX_BLOCKS*2];
	int bank_count = MAX_BLOCKS;
	int pointer_count = 0x8000;
	int pointer_count2 = 0x8000;
};

struct simple_string{
	int length = 0;
	char *data = nullptr;
	simple_string() = default;
	constexpr simple_string(const simple_string&) = default;

	simple_string &operator=(simple_string &&move)
	{
		delete []data;
		data = move.data;
		move.data = nullptr;
		length = move.length;
		return *this;
	}
	~simple_string()
	{
		delete []data;
	}
};

template <typename ...A>
void error(const char *message, A... args)
{
	printf(message, args...);
	exit(-1);
}

void double_click_exit()
{
	getc(stdin); //Pause before exit
}

FILE *open(const char *name, const char *mode)
{
	FILE *file = fopen(name, mode);
	if (!file) {
		error("Could not open \"%s\"\n", name);
	}
	return file;
}

int file_size(FILE *file)
{
	fseek(file, 0, SEEK_END);
	int size = ftell(file);
	fseek(file, 0, SEEK_SET);
	return size;
}

unsigned char *read_all(const char *file_name, bool text_mode = false, unsigned int minimum_size = 0)
{
	FILE *file = open(file_name, "rb");
	unsigned int size = file_size(file);
	unsigned char *file_data = new unsigned char[(size < minimum_size ? minimum_size : size) + (text_mode * 2)]();
	if (fread(file_data, 1, size, file) != size) {
		error("%s could not be fully read.  Please check file permissions.\n", file_name);
	}
	fclose(file);
	return file_data;
}

void write_all(unsigned char *data, const char *file_name, unsigned int size)
{
	FILE *file = open(file_name, "wb");
	if (fwrite(data, 1, size, file) != size) {
		error("%s could not be fully written.  Please check file permissions.\n", file_name);
	}
	fclose(file);
}

static const int sa1banks[8] = { 0 << 20, 1 << 20, -1, -1, 2 << 20, 3 << 20, -1, -1 };
struct ROM{
	enum class MapperType { lorom, sa1rom, fullsa1rom };

	unsigned char *data;
	unsigned char *real_data;
	char *name;
	bool sa1;
	int size;
	int header_size;
	MapperType mapper;

	void open(const char *n)
	{
		name = new char[strlen(n)+1]();
		strcpy(name, n);
		FILE *file = ::open(name, "r+b");	//call global open
		size = file_size(file);
		header_size = size & 0x7FFF;
		size -= header_size;
		data = read_all(name, false, MAX_ROM_SIZE + header_size);
		fclose(file);
		real_data = data + header_size;
		sa1 = real_data[0x7fd5] == 0x23;
		if (sa1) {
			if (real_data[0x7fd7] == 0x0D) {
				mapper = MapperType::fullsa1rom;
			} else {
				mapper = MapperType::sa1rom;
			}
		} else {
			mapper = MapperType::lorom;
		}
	}

	void close()
	{
		write_all(data, name, size + header_size);
		delete []data;
		delete []name;
	}

	int pc_to_snes(int address) {
		address -= header_size;

		if (mapper == MapperType::lorom) {
			return ((address << 1) & 0x7F0000) | (address & 0x7FFF) | 0x8000;
		} else if (mapper == MapperType::sa1rom) {
			for (int i = 0; i < 8; i++) {
				if (sa1banks[i] == (address & 0x700000)) {
					return 0x008000 | (i << 21) | ((address & 0x0F8000) << 1) | (address & 0x7FFF);
				}
			}
		} else if (mapper == MapperType::fullsa1rom) {
			if ((address & 0x400000) == 0x400000) {
				return address | 0xC00000;
			}
			if ((address & 0x600000) == 0x000000) {
				return ((address << 1) & 0x3F0000) | 0x8000 | (address & 0x7FFF);
			}
			if ((address & 0x600000) == 0x200000) {
				return 0x800000 | ((address << 1) & 0x3F0000) | 0x8000 | (address & 0x7FFF);
			}
		}
		return -1;
	}

	int snes_to_pc(int address) {
		if (mapper == MapperType::lorom) {
			if ((address & 0xFE0000) == 0x7E0000 || (address & 0x408000) == 0x000000 || (address & 0x708000) == 0x700000)
				return -1;
			address = (address & 0x7F0000) >> 1 | (address & 0x7FFF);
		} else if (mapper == MapperType::sa1rom) {
			if ((address & 0x408000) == 0x008000) {
				address = sa1banks[(address & 0xE00000) >> 21] | ((address & 0x1F0000) >> 1) | (address & 0x007FFF);
			} else if ((address & 0xC00000) == 0xC00000) {
				address = sa1banks[((address & 0x100000) >> 20) | ((address & 0x200000) >> 19)] | (address & 0x0FFFFF);
			} else {
				address = -1;
			}
		} else if (mapper == MapperType::fullsa1rom) {
			if ((address & 0xC00000) == 0xC00000) {
				address = (address & 0x3FFFFF) | 0x400000;
			} else if ((address & 0xC00000) == 0x000000 || (address & 0xC00000) == 0x800000) {
				if ((address & 0x008000) == 0x000000)
					return -1;
				address = (address & 0x800000) >> 2 | (address & 0x3F0000) >> 1 | (address & 0x7FFF);
			} else {
				return -1;
			}
		} else {
			return -1;
		}

		return address + header_size;
	}
};

simple_string get_line(const char *text, int offset) {
	simple_string string;
	if (!text[offset]) {
		return string;
	}
	string.length = strcspn(text+offset, "\r\n")+1;
	string.data = new char[string.length]();
	strncpy(string.data, text+offset, string.length-1);
	return string;
}

char *trim(char *text)
{
	while (isspace(*text)) {
		text++;
	}
	for (int i = strlen(text); isspace(text[i-1]); i--) {	//trim back
		text[i] = 0;
	}
	return text;
}

unsigned char *binary_strstr(unsigned char *data, int data_length, const char *search_for)
{
	int search_length = strlen(search_for);
	for (int i = 0; i < data_length - search_length; i++) {
		if (!memcmp(data+i, search_for, search_length)) {
			return data+i;
		}
	}
	return nullptr;
}

#define ERROR(S) delete []list_data; delete []block_list; error(S, line_number)

#define MIN(a,b) (((a)<(b))?(a):(b))
#define MAX(a,b) (((a)>(b))?(a):(b))

char* populate_block_list(block *block_list, const char *list_data)
{
	bool dsc = false;
	int line_number = 0, i = 0, bytes_read;
	short acts_like;
	simple_string current_line;
	do {
		current_line = static_cast<simple_string &&>(get_line(list_data, i));
		i += current_line.length;
		line_number++;

		if (!current_line.length || !trim(current_line.data)[0] || current_line.data[0] == ';') continue;

		if (strcmp(current_line.data, "@dsc") == 0) {
			dsc = true;
			break;
		}

		int *block_ids = new int[1];
		int num1,num2,temp;
		char c;

		acts_like = -1;

		if (current_line.data[0] == 'R') {
			if (!sscanf(current_line.data, "%*c%x%c%x%n", &num1, &c, &num2, &bytes_read) || c != '-') {
				ERROR("Error in block list on line %d: Invalid line start.\n");
			}
			// Get all blocks inside rectangle into block_id array
			temp = MIN(num1 & 0xFFF0, num2 & 0xFFF0) | MIN(num1 & 0xF, num2 & 0xF);
			num2 = MAX(num1 & 0xFFF0, num2 & 0xFFF0) | MAX(num1 & 0xF, num2 & 0xF);
			num1 = temp;

			if (num1 < MIN_BLOCKS || num2 >= MAX_BLOCKS) {
				ERROR("Error in block list on line %d: Block id out of bounds.\n");
			}

			int width,height;
			height = ((((num2 & 0xFFF0) - (num1 & 0xFFF0)) >> 4) + 1);
			width = (((num2 & 0xF) - (num1 & 0xF)) + 1);
			num2 = num1 & 0xFFF0;
			num1 = num1 & 0xF;

			temp = width * height;
			block_ids = new int[temp];
			for (int j = 0; j < temp; j++)
				block_ids[j] = (num2 + ((j / width) << 4)) | (num1 + (j % width));

			if (current_line.data[bytes_read] == ':')
				sscanf(current_line.data, "%*c%*x%*c%*x%*c%hx%n", &acts_like, &bytes_read);
		} else {
			if (!sscanf(current_line.data, "%x%c%n", &num1, &c, &bytes_read)) {
				ERROR("Error in block list on line %d: Invalid line start.\n");
			}
			if (c == '-') {
				if (!sscanf(current_line.data, "%*x%*c%x%n", &num2, &bytes_read)) {
					ERROR("Error in block list on line %d: Invalid line start.\n");
				}
				// Get all blocks in range into block_id array
				temp = MAX(num1, num2);
				num1 = MIN(num1, num2);
				num2 = temp;

				if (num1 < MIN_BLOCKS || num2 >= MAX_BLOCKS) {
					ERROR("Error in block list on line %d: Block id out of bounds.\n");
				}

				temp = num2-num1+1;
				block_ids = new int[temp];
				for (int j = num1; j <= num2; j++) {
					block_ids[j-num1] = j;
				}

				if (current_line.data[bytes_read] == ':') {
					sscanf(current_line.data, "%*x%*c%*x%*c%hx%n", &acts_like, &bytes_read);
				}
			} else {
				bytes_read--;
				// Get block into block_id array
				if (num1 < MIN_BLOCKS || num1 >= MAX_BLOCKS) {
					ERROR("Error in block list on line %d: Block id out of bounds.\n");
				}

				temp = 1;
				block_ids[0] = num1;

				if (current_line.data[bytes_read] == ':') {
					sscanf(current_line.data, "%*x%*c%hx%n", &acts_like, &bytes_read);
				}
			}
		}

		if (acts_like >= MAX_BLOCKS) {
			ERROR("Error in block list on line %d: Acts like out of bounds.\n");
		}

		if (isspace(current_line.data[bytes_read])) {
			char *file_name = trim(current_line.data + bytes_read);

			for (int j = 0; j < temp; j++) {
				block_list[block_ids[j]].file_name = new char[strlen(file_name) + 1];
				strcpy(block_list[block_ids[j]].file_name, file_name);
				block_list[block_ids[j]].line = line_number;
				if (acts_like != -1) {
					block_list[block_ids[j]].acts_like = acts_like;
				}
				if (!block_list[block_ids[j]].file_name[0]) {
					ERROR("Error in block list on line %d: Missing filename.\n");
				}
			}
		} else {
			ERROR("Error in block list on line %d: Missing space or acts like seperator.\n");
		}
		delete[] block_ids;
	} while (current_line.length);

	if (dsc && (signed)(strlen(list_data) - i) > 0) {
		char *dsc_data = new char[strlen(list_data) - i];
		strcpy(dsc_data, 1 + list_data + i);
		delete[] list_data;
		return dsc_data;
	}

	delete []list_data;
	return nullptr;
}

#undef ERROR

#undef MIN
#undef MAX

bool patch(const char *patch_name, ROM &rom, bool debug_flag, bool show_warnings, const char *debug_name)
{
	if (!asar_patch(patch_name, (char *)rom.real_data, MAX_ROM_SIZE, &rom.size)) {
		int error_count;
		const errordata *errors = asar_geterrors(&error_count);
		error("An error has been detected:\n%s\n", errors->fullerrdata);
	}
	int count = 0;
	if (show_warnings) {
		const errordata * const warn_data = asar_getwarnings(&count);
		for (int x = 0; x < count; x++) {
			if (warn_data[x].errid != 1001) { // Relative path to patch file
				printf("%s\n", warn_data[x].fullerrdata);
			}
		}
	}
	if (debug_flag) {
		printf("__________________________________\n");
		const labeldata *labels = asar_getalllabels(&count);
		printf("\nfile \"%s\": \n\nlabel count: %d\n\n", debug_name, count);
		for (int x = 0; x < count; x++) {
			printf("%s: $%X\n", labels[x].name, labels[x].location);
		}
		const char * const *print_data = asar_getprints(&count);
		printf("\nprint count: %d\n\n", count);
		for (int x = 0; x < count; x++) {
			printf("print %d: %s\n", x+1, print_data[x]);
		}
		printf("__________________________________\n");
	}
	return true;
}

int get_pointer(unsigned char *data, int address, int size = 3, int bank = 0x00)
{
	address = (data[address])
		| (data[address + 1] << 8)
		| ((data[address + 2] << 16) * (size-2));
	return address | (bank << 16);
}

void write_block(block &current, int i, block_table_data &block_data, FILE *dsc)
{
	block_data.banks[i] = (current.pointer & 0xFF0000) >> 16;
	block_data.pointers[i*2] = current.pointer & 0xFF;
	block_data.pointers[i*2+1] = (current.pointer & 0xFF00) >> 8;
	if (current.acts_like != -1) {
		block_data.acts_likes[i*2] = (current.acts_like & 0x0000FF);
		block_data.acts_likes[i*2+1] = (current.acts_like & 0x00FF00) >> 8;
	}

	int transparency = 0;
	if (current.editordisplay.transparent) {
		transparency = 8;
	}
	if (current.editordisplay.description != nullptr) {
		fprintf(dsc, "%x\t%x\t%s\n", i, transparency, current.editordisplay.description);
	} else {
		fprintf(dsc, "%x\t%x\t%s\n", i, transparency, current.file_name);
	}
	if (current.editordisplay.pswitch != -MAX_BLOCKS) {
		int tile = current.editordisplay.pswitch;
		if (current.editordisplay.pswitch_relative) {
			tile += i;
		}
		if (tile < 0 || tile >= MAX_BLOCKS) {
			printf("\nWarning: P-Switch map16 for block %X out of range (%X)", i, tile);
		}
		fprintf(dsc, "%x\t4\t%x\n", i, tile);
	}
	if (current.editordisplay.hidden != -MAX_BLOCKS) {
		int tile = current.editordisplay.hidden;
		if (current.editordisplay.hidden_relative) {
			tile += i;
		}
		if (tile < 0 || tile >= MAX_BLOCKS) {
			printf("\nWarning: Hidden map16 for block %X out of range (%X)", i, tile);
		}
		fprintf(dsc, "%x\t2\t%x\n", i, tile);
	}
	if (current.editordisplay.sprite != -MAX_BLOCKS) {
		if (current.editordisplay.sprite < 0 || current.editordisplay.sprite >= 0x2400) {
			printf("\nWarning: Sprite map16 (content) for block %X out of range (%X)", i, current.editordisplay.sprite);
		}
		fprintf(dsc, "%x\t10\t%x\n", i, current.editordisplay.sprite);
	}
}

bool duplicate_block(block *block_list, int i)
{
	for (int j = i - 1; j >= 0; j--) {
		if (block_list[j].line && !strcmp(block_list[i].file_name, block_list[j].file_name)) {
			block_list[i].pointer = block_list[j].pointer;
			block_list[i].editordisplay = block_list[j].editordisplay;
//			block_list[i].description = block_list[j].description;
			return true;
		}
	}
	return false;
}

void clean_old_blocks(block_table_data &block_data, ROM &rom, int pointer_offset, int new_routine_offset, int pointer_offset2 = -1, int bank_offset = -1)
{
	FILE *block_clean = open("block_clean.asm", "w");
	for (int i = 0; i < MAX_BLOCKS; i++) {
		if (block_data.banks[i]) {
			int pointer = get_pointer(block_data.pointers, i*2, 2, block_data.banks[i]);
			fprintf(block_clean, ";cleaning block 0x%X\nautoclean $%X\n", i, pointer);
		}
	}

	fprintf(block_clean, ";cleaning pointer pages 00-3F\nautoclean $%X\n", pointer_offset);
	if (pointer_offset2 != -1) {
		fprintf(block_clean, ";cleaning pointer pages 40-7F\nautoclean $%X\n", pointer_offset2);
		fprintf(block_clean, ";cleaning bank pointer\nautoclean $%X\n", bank_offset);
	}

	//block shared routines
	for (int i = 0; i < 100; i++) {
		int pointer = get_pointer(rom.data, rom.snes_to_pc(0x0CB66E + new_routine_offset + i * 3));
		if (pointer != 0xFFFFFF) {
			fprintf(block_clean, ";cleaning routine 0x%X\nautoclean $%X\n", i, pointer);
			fprintf(block_clean, "org $%X\n", 0x0CB66E + new_routine_offset + i * 3);
			fprintf(block_clean, "dl $FFFFFF\n");
		}
	}
	fclose(block_clean);
	patch("block_clean.asm", rom, false, false, "block_clean.asm"); //has no labels to provide debug info.
}

void clean_btsd(unsigned char *offset, ROM &rom)
{
	printf("BTSD detected, attempting removal.\n");
	unsigned short version = (offset[0x17] << 8) + offset[0x18];
	if (version != 0x3136) {
		error("BTSD hack version %X not supported.\n", version);
	}

	FILE *block_clean = open("block_clean.asm", "w");
	unsigned int pointer_address = rom.snes_to_pc((offset[0x19] << 16) | 0x8000);
	for (int i = 0; i < 0x4000; i++) {
		int pointer = offset[0x1A + i] << 16;
		if (pointer) {
			pointer |= rom.data[pointer_address + i];
			pointer |= (rom.data[pointer_address + 0x4000 + i] << 8);
			fprintf(block_clean, "autoclean $%X\n", pointer);
		}
    }
    fprintf(block_clean, "autoclean $%X\n", rom.pc_to_snes((int)(offset - rom.data)));
    fprintf(block_clean, "autoclean $%X\n", rom.pc_to_snes(pointer_address));
	fclose(block_clean);
	patch("block_clean.asm", rom, false, false, "block_clean.asm"); //has no labels to provide debug info.
	printf("BTSD successfully removed.\n");
}

void clean_hack(ROM &rom, block_table_data &block_data)
{
	if (rom.data[rom.snes_to_pc(0x06F690)] == 0x8B) {		//already installed load old tables
		unsigned char *base = binary_strstr(rom.real_data, rom.size, "GPS_VeRsIoN");
		if (!base) {		//Can't find GPS. VERY old versions of GPS might trigger this but it's safer to assume it's something else.
			error("Error: Unidentified custom block code detected\n");
		} else {		// Clean GPS
			char version = (base[11] == 1) ? 0 : 1; // 0 - Base ASM Version 1 , 1 - Base ASM Version 2 and above

			int bank_offset, pointer_offset;
			bank_offset = get_pointer(base, 12+version);
			pointer_offset = get_pointer(base, 15+version);
			block_data.bank_count = get_pointer(base, 18+version, 2); //kinda hacky, but makes sense(ish)
			block_data.pointer_count = get_pointer(base, 20+version, 2); //kinda hacky, but makes sense(ish)
			memcpy(block_data.banks, rom.data+rom.snes_to_pc(bank_offset), block_data.bank_count);
			memcpy(block_data.pointers, rom.data+rom.snes_to_pc(pointer_offset), block_data.pointer_count);

			if (version == 1 && base[12] == 1) {	// If pages 40-7F contain data remove these too
				int pointer_offset2 = get_pointer(base, 23);
				block_data.pointer_count2 = get_pointer(base, 26, 2); //kinda hacky, but makes sense(ish)
				memcpy(block_data.pointers+0x8000, rom.data+rom.snes_to_pc(pointer_offset2), block_data.pointer_count2);
				clean_old_blocks(block_data, rom, pointer_offset, (base[11] >= 3) ? 1 : 0, pointer_offset2, bank_offset);
			} else {		// Otherwise just clean the currently known blocks
				clean_old_blocks(block_data, rom, pointer_offset, (base[11] >= 3) ? 1 : 0);
			}
		}
		memset(block_data.banks, 0, MAX_BLOCKS);	//clear all old pointers
	} else { //check for BTSD
		unsigned char *btsd_offset = binary_strstr(rom.real_data, rom.size, "Blocktool Super Deluxe");
		if (btsd_offset) {
			clean_btsd(btsd_offset, rom);
		}
	}
}

void create_shared_patch(char *routine_path, ROM &rom, bool debug_flag, bool show_warnings)
{
	FILE *shared_patch = open("shared.asm", "w");
	fprintf(shared_patch,
				"macro include_once(target, base, offset)\n"
				"	if !<base> != 1\n"
				"		!<base> = 1\n"
				"		pushpc\n"
				"		if read3(<offset>*3+$0CB66F) != $FFFFFF\n"
				"			<base> = read3(<offset>*3+$0CB66F)\n"
				"		else\n"
				"			freecode cleaned\n"
				"			#<base>:\n"
//				"			namespace <base>\n" // namespaces aren't needed anymore, routines use macro labels only
				"			incsrc \"<target>\"\n"
//				"			namespace off\n"
				"			ORG <offset>*3+$0CB66F\n"
				"			dl <base>\n"
				"		endif\n"
				"		pullpc\n"
				"	endif\n"
				"endmacro\n");
	DIR *routine_directory = opendir(routine_path ? routine_path : "routines/");
	dirent *routine_file = nullptr;
	if (!routine_directory) {
		error("Unable to open the routine directory \"%s\"\n", routine_path ? routine_path : "routines/");
	}
	int routine_count = 0;
	while ((routine_file = readdir(routine_directory)) != NULL) {
		char *name = routine_file->d_name;
		if (!strcmp(".asm", name + strlen(name) - 4)) {
			if (routine_count > 100) {
				closedir(routine_directory);
				error("More than 100 routines located.  Please remove some.\n", "");
			}
			name[strlen(name) - 4] = 0;
			fprintf(shared_patch, 	"!%s = 0\n"
						"macro %s()\n"
						"\t%%include_once(\"%s%s.asm\", %s, $%.2X)\n"
						"\tJSL %s\n"
						"endmacro\n",
						name, name, routine_path ? routine_path : "routines/",
						name, name, routine_count, name);
			routine_count++;
		}
	}
	closedir(routine_directory);
	printf("%d Shared routines registered in \"%s\"\n", routine_count, routine_path ? routine_path : "routines/");
	fclose(shared_patch);
}

int assemble_blocks(ROM &rom, block *block_list, block_table_data &block_data, char *block_path, int &largest_block_id_hi, char* dsc_data, bool debug_flag, bool show_warnings)
{
	char dsc_name[FILENAME_MAX] = {0};
	strcpy(strcpy(dsc_name, rom.name) + strlen(dsc_name)-3, "dsc");
	FILE *block_patch = open("block_boilerplate.asm", "w");
	FILE *dsc = open(dsc_name, "w");
	int largest_block_id_lo = 0;

	for (int i = 0; i < MAX_BLOCKS; i++) {
		if (block_list[i].file_name) {
			if (duplicate_block(block_list, i)) {
				write_block(block_list[i], i, block_data, dsc);
				if (i >= 0x4000) {
					largest_block_id_hi = i - 0x3FFF;
				} else {
					largest_block_id_lo = i;
				}
				continue;
			}

			if (!freopen("block_boilerplate.asm", "w", block_patch)) {
				error("Couldn't open block_boilerplate.asm\n");
			}

			fprintf(block_patch, "incsrc \"defines.asm\"\n"
					     "incsrc \"shared.asm\"\n"
					     "freecode cleaned\n"
					     "_BLOCK_ENTRY_:\n"
					     "incsrc \"%s%s\"\n"
						 "_BLOCK_EXIT_:\n",
					     block_path ? block_path : "blocks/", block_list[i].file_name);
			fflush(block_patch);
			patch("block_boilerplate.asm", rom, debug_flag, show_warnings, block_list[i].file_name);
			int exit_pointer = asar_getlabelval("_BLOCK_EXIT_");
			block_list[i].pointer = asar_getlabelval("_BLOCK_ENTRY_");
			int size = exit_pointer - block_list[i].pointer;
			if (size < 0) {
				error("Error: block %X (file: %s) has a negative size. Are you trying to insert a patch?\n", i, block_list[i].file_name);
			}
			if (size > 0x8000) {
				error("Error: block %X (file: %s) exceeds the size of a bank. Are you trying to insert a patch?\n", i, block_list[i].file_name);
			}
			if (!(rom.data[rom.snes_to_pc(block_list[i].pointer)] == 0x42 || rom.data[rom.snes_to_pc(block_list[i].pointer)] == 0x37)) {
				error("Error: block %X (file: %s) lacks a db $42 or db $37 header.\n", i, block_list[i].file_name);
			}

			int count;
			const char * const *prints = asar_getprints(&count);
			if (count > 0) {
				char* desc = (char*)malloc(strlen(prints[0])+1);
				strcpy(desc, prints[0]);
				block_list[i].editordisplay.description = desc;
				for (int print = 1; print < count; print++) {
					int tile = -1, sign = 1;
					if (strncmp(prints[print], "@pswitch=", strlen("@pswitch=")) == 0) {
						const char* number = prints[print]+strlen("@pswitch=");
						if (number[0] == '+' || number[0] == '-') {
							if (number[0] == '-') {
								sign = -1;
							}
							block_list[i].editordisplay.pswitch_relative = true;
							number += 1;
						}
						if (sscanf(number, "%x", &tile) == 1) {
							block_list[i].editordisplay.pswitch = tile*sign;
						}
					} else if (strncmp(prints[print], "@hidden=", strlen("@hidden=")) == 0) {
						const char* number = prints[print]+strlen("@hidden=");
						if (number[0] == '+' || number[0] == '-') {
							if (number[0] == '-') {
								sign = -1;
							}
							block_list[i].editordisplay.hidden_relative = true;
							number += 1;
						}
						if (sscanf(number, "%x", &tile) == 1) {
							block_list[i].editordisplay.hidden = tile*sign;
						}
					} else if (strncmp(prints[print], "@content=", strlen("@content=")) == 0) {
						const char* number = prints[print]+strlen("@content=");
						if (sscanf(number, "%x", &tile) == 1) {
							block_list[i].editordisplay.sprite = tile;
						}
					} else if (strcmp(prints[print], "@transparent") == 0) {
						block_list[i].editordisplay.transparent = true;
					}
				}
			} else {
				block_list[i].editordisplay.description = nullptr;
			}

			write_block(block_list[i], i, block_data, dsc);
			if (i >= 0x4000) {
				largest_block_id_hi = i - 0x3FFF;
			} else {
				largest_block_id_lo = i;
			}
		}
	}

	if (dsc_data != nullptr) {
		fprintf(dsc, "%s", dsc_data);
	}
	fclose(dsc);
	fclose(block_patch);
	return largest_block_id_lo + 1;
}

int main(int argc, char *argv[])
{
	bool debug_flag = false;
	char *block_path = nullptr;
	char *routine_path = nullptr;
	char list[FILENAME_MAX] = "list.txt";
	bool O1 = false, O2 = false;
	bool keep_temp = false;
	bool ask_for_rom = false;
	bool show_warnings = true;
	ROM rom;

    if (argc < 2) {
        atexit(double_click_exit);
    }

	printf("Gopher Popcorn Stew Version " GPS_EXE_VERSION "\n\n");

	if (!asar_init()) {
		error("Error: Outdated or missing Asar library, please redownload the tool.\n", "");
	}

	for (int i = 1; i < argc; i++) {
		if (!strcmp(argv[i], "-h") || !strcmp(argv[i], "--help") ) {
			printf("Usage: GPS <options> <ROM>\nOptions are:\n");
			printf("-d\t\tEnable debug output\n");
			printf("-k\t\tKeep debug files\n");
			printf("-nw\t\tHide Asar warnings\n");
			printf("-l <listpath>\tSpecify a custom list file (Default: %s)\n", list);
			printf("-b <blockpath>\tSpecify a custom block directory (Default blocks/)\n");
			printf("-s <sharedpath>\tSpecify a shared routine directory (Default routines/)\n");
			printf("-O1\t\tEnable optimization level 1, reduces pointer table size when possible\n");
			printf("-O2\t\tEnable optimization level 2, reduces bank byte size when possible\n");
			printf("-rom\t\tAsks for the rom even when passing command line parameters\n");
			printf("Please read the readme before using optimization.\n");
			exit(0);
		} else if (!strcmp(argv[i], "-d") || !strcmp(argv[i], "--debug")) {
			debug_flag = true;
		} else if (!strcmp(argv[i], "-k")) {
			keep_temp = true;
		} else if (!strcmp(argv[i], "-nw")) {
			show_warnings = false;
		} else if (!strcmp(argv[i], "-O1")) {
			O1 = true;
		} else if (!strcmp(argv[i], "-O2")) {
			O1 = O2 = true;
		} else if (!strcmp(argv[i], "-O3")) {
			error("Option \"%s\" not implemented.\n", argv[i]);
		} else if (!strcmp(argv[i], "-rom")) {
			ask_for_rom = true;
		} else if (!strcmp(argv[i], "-b") && i < argc - 2) {
			block_path = argv[i+1];
			i++;
		} else if (!strcmp(argv[i], "-s") && i < argc - 2) {
			routine_path = argv[i+1];
			i++;
		} else if (!strcmp(argv[i], "-l") && i < argc - 2) {
			strncpy(list, argv[i+1], FILENAME_MAX);
			i++;
		} else {
			if (i == argc-1) {
				break;
			}
			error("Error: Invalid command line option \"%s\".\n", argv[i]);
		}
	}

	if (argc < 2 || ask_for_rom) {
		printf("Enter a ROM file name, or drag and drop the ROM here: ");
		char ROM_name[FILENAME_MAX];
		if (fgets(ROM_name, FILENAME_MAX, stdin)) {
			int length = strlen(ROM_name)-1;
			ROM_name[length] = 0;
			if ((ROM_name[0] == '"' && ROM_name[length - 1] == '"') ||
			   (ROM_name[0] == '\'' && ROM_name[length - 1] == '\'')) {
				ROM_name[length - 1] = 0;
				for (int i = 0; ROM_name[i]; i++) {
					ROM_name[i] = ROM_name[i+1]; //no buffer overflow there are two null chars.
				}
			}
		}
		rom.open(ROM_name);
	} else {
		rom.open(argv[argc-1]);
	}

	if (debug_flag) {
		printf("ROM Information:\nSA-1: %d\nROM Size: 0x%X\nHeader Size: 0x%X\n\n", rom.sa1, rom.size, rom.header_size);
	}

	int acts_like_pointer = get_pointer(rom.data, rom.snes_to_pc(0x06F624));
	if (acts_like_pointer == 0xFFFFFF) {
		error("Please save a level in Lunar Magic before using GPS\n", "");
	}

	block *block_list = new block[MAX_BLOCKS];
	char* dsc_data = populate_block_list(block_list, (char *)read_all(list, true));

	block_table_data block_data;
	if (debug_flag) {
		printf("Acts like data at 0x%X ($%X)\n", rom.snes_to_pc(acts_like_pointer), acts_like_pointer);
	}
	memcpy(block_data.acts_likes, rom.data+rom.snes_to_pc(acts_like_pointer), 0x8000);

	if (rom.data[rom.snes_to_pc(0x06F63C)] != 0xFF) {
		acts_like_pointer = get_pointer(rom.data, rom.snes_to_pc(0x06F63A)) + 0x8000;	// acts_like_pointer now points to the acts like for pages 40+
		if (debug_flag) {
			printf("Acts like data at 0x%X ($%X)\n", rom.snes_to_pc(acts_like_pointer), acts_like_pointer);
		}
		memcpy(block_data.acts_likes + 0x8000, rom.data+rom.snes_to_pc(acts_like_pointer), 0x8000);
	}

	clean_hack(rom, block_data);
	create_shared_patch(routine_path, rom, debug_flag, show_warnings);

	int largest_block_id_hi = -1;	// largest block id of pages 40+. largest_block_id_lo contains the largest block for pages 00-3F
	int largest_block_id_lo = assemble_blocks(rom, block_list, block_data, block_path, largest_block_id_hi, dsc_data, debug_flag, show_warnings);

	if (largest_block_id_hi != -1) {
		if (rom.data[rom.snes_to_pc(0x06F63C)] == 0xFF) {
			error("Cannot insert custom blocks on pages 40+\nPlease make sure to edit at least one block in that area to use it\n");
		}
		write_all(block_data.pointers           , "__pointers_1.bin"  , O1 ? largest_block_id_lo : 0x8000);
		write_all(block_data.pointers   + 0x8000, "__pointers_2.bin"  , O1 ? largest_block_id_hi * 2 : 0x8000);
		write_all(block_data.banks              , "__banks.bin"       , O2 ? (largest_block_id_hi + 0x4000) : 0x8000);
		write_all(block_data.acts_likes         , "__acts_likes_1.bin", 0x8000); // O3 was here originally. I removed it since it's pretty much pointless even if it were to be implemented.
		write_all(block_data.acts_likes + 0x8000, "__acts_likes_2.bin", 0x8000); // (^)
	} else {
		write_all(block_data.banks,      "__banks.bin",        O2 ? largest_block_id_lo : 0x4000);
		write_all(block_data.pointers,   "__pointers_1.bin",   O1 ? largest_block_id_lo * 2 : 0x8000);
		write_all(block_data.acts_likes, "__acts_likes_1.bin", 0x8000);
	}

	FILE *settings = open("__temp_settings.asm", "w");
	fprintf(settings, "!__insert_pages_40_plus = %d", (largest_block_id_hi == -1) ? 0 : 1);
	fclose(settings);

	patch("main.asm", rom, debug_flag, show_warnings, "main.asm");

	if (!keep_temp) {
		remove("__banks.bin");
		remove("__pointers_1.bin");
		remove("__pointers_2.bin");
		remove("__acts_likes_1.bin");
		remove("__acts_likes_2.bin");
		remove("__temp_settings.asm");
		remove("shared.asm");
		remove("block_clean.asm");
		remove("block_boilerplate.asm");
	}

	unsigned char* version = binary_strstr(rom.real_data, rom.size, "GPS_VeRsIoN")+11;
	if (*version != GPS_ASM_VERSION) {
		printf("Warning: Installed ASM version %d does not match expected version %d\nPlease make sure the base asm is up to date with the tool\nDo you want to apply changes anyway? (y/n) (Default: n) ", (*version), GPS_ASM_VERSION);
		if (getc(stdin) != 'y') {
			error("No Changes Applied\n");
		}
	}

	rom.close();
	asar_close();
	delete []block_list;

	printf("\nAll blocks applied successfully!\n");
	return 0;
}
